﻿/**
© Florian Joncour, 2013-2018 florian@zetta-sys.com

Ce logiciel est un programme informatique faisant interface à la bibliothèque
Perspective3D, un outil de modélisation 3D à partir de vues orthographiques 2D.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site "http://www.cecill.info".

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée.  Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme,  le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard  l'attention de l'utilisateur est attirée sur les risques
associés au chargement,  à l'utilisation,  à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant  des  connaissances  informatiques approfondies.  Les
utilisateurs sont donc invités à charger  et  tester  l'adéquation  du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.
**/

#include <stdio.h>
#include <iostream>
#include "mainwindow.h"
#include <QApplication>

#include <QDebug>

#include <QTextCodec>
#include <QTranslator>
#include <QLibraryInfo>
#include <QStyleFactory>
#include <QIcon>
#include <QSurfaceFormat>

#ifdef Q_OS_WIN
//    #include <QWindowsWindowFunctions>
#endif // Q_OS_WIN

//#include <QSurfaceFormat>

#include "perspective_types.h"
#include "automate.h"
#include "version.h"
#include "Qt/Conversion.h"

#ifdef DEBUG
    #include <QLoggingCategory>

    #include "perspective_sys.h"
    #ifdef Q_OS_UNIX
        #include "errno.h"
        #include "error.h"
        #include <sys/resource.h>
    #endif //Q_OS_UNIX

#endif // DEBUG

#include "style.h"

#ifdef DEBUG
void GestionnaireMessages(QtMsgType type, const QMessageLogContext &context, const QString &msg)
{
    QByteArray localMsg = msg.toLocal8Bit();
    switch (int(type))
    {
        case QtInfoMsg:
        case QtDebugMsg:
            fprintf(stdout, "file://%s:%u (Information) : %s\n", context.file, context.line, localMsg.constData());
            break;
        case QtWarningMsg:
            fprintf(stderr, "file://%s:%u (Attention) : %s\n", context.file, context.line, localMsg.constData());
            break;
        case QtCriticalMsg:
            fprintf(stderr, "file://%s:%u (Erreur critique) : %s\n", context.file, context.line, localMsg.constData());
            break;
        case QtFatalMsg:
            fprintf(stderr, "file://%s:%u (Erreur fatale) : %s\n", context.file, context.line, localMsg.constData());
            abort();
    }
}
#endif // DEBUG

int main(int argc, char *argv[])
{
//#ifdef DEBUG
//    qInstallMessageHandler(GestionnaireMessages);
//#endif

#ifdef DEBUG
//    qSetMessagePattern("[%{appname}] %{if-category}(%{category})%{endif} %{file}:%{line} :: %{message}");
    qSetMessagePattern("[%{appname}] %{if-category}(%{category})%{endif} :: %{message}");

    QLoggingCategory::setFilterRules("qt.network.ssl.warning=false");
#endif // DEBUG

#ifdef DEBUG
#ifdef PSYS_LINUX
    struct rlimit core_limit;
    core_limit.rlim_cur = RLIM_INFINITY;
    core_limit.rlim_max = RLIM_INFINITY;

    if (setrlimit(RLIMIT_CORE, &core_limit) < 0)
        std::cerr << "setrlimit: " << "Attention, impossible de changer rlimit pour générer les dumps mémoire: " << strerror(errno) << std::endl;
    else
    {
        std::cout << "setrlimit fixé !" << std::endl;
    }
#endif // __linux__
#endif // DEBUG

#if (QT_VERSION >= QT_VERSION_CHECK(5, 0, 0))
    QTextCodec::setCodecForLocale(QTextCodec::codecForName("UTF-8"));
#else
    QTextCodec::setCodecForCStrings(QTextCodec::codecForName("UTF-8"));
#endif

    QSurfaceFormat format;
    format.setDepthBufferSize(24);
    format.setRenderableType(QSurfaceFormat::OpenGL);
#ifdef DEBUG
    format.setProfile(QSurfaceFormat::CompatibilityProfile);

//    format.setProfile(QSurfaceFormat::CoreProfile);
//    format.setVersion(3, 2);

//    format.setProfile(QSurfaceFormat::CoreProfile);
//    format.setVersion(4, 1);

    format.setOptions(QSurfaceFormat::DebugContext);
//    format.setOptions(QSurfaceFormat::DebugContext | QSurfaceFormat::DeprecatedFunctions);
#else
    format.setProfile(QSurfaceFormat::CompatibilityProfile);
#endif
    QSurfaceFormat::setDefaultFormat(format);

//    QCoreApplication::setAttribute(Qt::AA_UseDesktopOpenGL,true);

//    QCoreApplication::setAttribute(Qt::AA_UseOpenGLES,true);
//    QCoreApplication::setAttribute(Qt::AA_UseDesktopOpenGL,false);

    //    QCoreApplication::setAttribute(Qt::AA_UseSoftwareOpenGL,true);
//    QCoreApplication::setAttribute(Qt::AA_ForceRasterWidgets,true);

    QApplication app(argc, argv);
    app.setApplicationName(VER_PRODUCTNAME_STR);
    app.setApplicationVersion(VER_PRODUCTVERSION_STR);
    app.setOrganizationName(VER_COMPANYNAME_STR);
    app.setOrganizationDomain(VER_COMPANYDOMAIN_STR);
    app.setWindowIcon(QIcon(GenLogoPerspective3D(true)));

//    QDir chemin_zetta6_d(argv[0]);
//    chemin_zetta6_d.cdUp();
//    QString chemin_zetta6(chemin_zetta6_d.absolutePath());
    QString chemin_zetta6 = app.applicationDirPath();

//    QSurfaceFormat format;
//    format.setDepthBufferSize(24);
//    format.setStencilBufferSize(8);
//    format.setVersion(3, 2);
//    format.setProfile(QSurfaceFormat::CoreProfile);
//    QSurfaceFormat::setDefaultFormat(format);

    /* Gestion de la langue: */
    QString locale = QLocale::system().name().section('_', 0, 0); // Langue du système (pour test: $LC_ALL=C).
//    QString locale("es"); // Force une traduction.

    /* Traductions internes Qt: */
    QTranslator qtTranslator;
    qtTranslator.load("qt_" + QLocale::system().name(), QLibraryInfo::location(QLibraryInfo::TranslationsPath));
    app.installTranslator(&qtTranslator);

    Perspective3D::i18n::lang_p3d_t lang_p3d = P3D_LANG_DEFAUT;
    QTranslator translator;

    if (locale == QString("fr"))
    {
        lang_p3d = Perspective3D::i18n::lang_p3d_t::P3D_LANG_FR;
    }
    else
    {
        if (!translator.load(chemin_zetta6 + QDir::separator() + QString("i18n") + QDir::separator() + QString("zetta6_") + locale)) // Erreur lors de la recherche de la traduction.
        {
            translator.load(chemin_zetta6 + QDir::separator() + QString("i18n") + QDir::separator() + QString("zetta6_en"));
        }
        app.installTranslator(&translator);
        lang_p3d = Perspective3D::i18n::lang_p3d_t::P3D_LANG_EN;
    }

    /* Gestion des arguments de la commande de lancement: */
    bool gen_auto = true; /* Permet la génération des modèles dès le lancement (si false, empêche la génération automatique même si c'est dans les paramètres du programme) */

    bool style = true;

    int debut_str_plan = 0;
    int fin_str_plan = 0;
    bool cassure_str = false;
    bool assigne_style = false;

    QStringList arguments = app.arguments();
    const unsigned int taille_args = arguments.size();
    for(unsigned int i=1; i<taille_args; ++i)
    {
        const QString &arg = arguments[i];
        if (arg.compare("-version", Qt::CaseInsensitive) == 0) // Version de Zetta6
        {
//            qDebug() << VER_PRODUCTVERSION_STR;
            std::cout << VER_PRODUCTVERSION_STR << std::endl;
            exit(0);
        }
        if (arg.compare("-dauto", Qt::CaseInsensitive) == 0) // Désactive la génération automatique du modèle malgré la configuration utilisateur.
        {
            gen_auto = false;
        }
        else if (arg.compare("-ns", Qt::CaseInsensitive) == 0) // Désactive l'utilisation du style pour l'application.
        {
            style = false;
        }
        else if (arg.compare("-styles", Qt::CaseInsensitive) == 0) // Liste des styles
        {
            qDebug() << QObject::tr("Styles") << ":" << QStyleFactory::keys();
            exit(0);
        }
        else if (arg.compare("-s", Qt::CaseInsensitive) == 0)
        {
            if ((i+1) < taille_args)
            {
#ifdef DEBUG
                qDebug() << "Assignation du style : " << arguments[i+1];
#endif
                app.setStyle(QStyleFactory::create(arguments[i+1]));
                ++i;
                assigne_style = true;
                continue;
            }
        }
//        else if (arg.contains(".dxf", Qt::CaseInsensitive)) // En dernier
        else
        {
            /* Recherche si il y a un plan à ouvrir. Accepte les espaces en ayant plusieurs arguments qui se suivent à
                condition que chacun contienne un caractère de séparation, comme '/' sous Unix par exemple. */

            if (arg.contains(QDir::separator()))
            {
                if (cassure_str)
                {
                    debut_str_plan = fin_str_plan = i;
                }
                else
                {
                    if (!debut_str_plan)
                    {
                        debut_str_plan = i;
                    }
                    fin_str_plan = i;
                }
            }
            else
            {
                if (debut_str_plan && fin_str_plan) // Il y a une cassure...
                {
                    cassure_str = true;
                }
            }
        }
    }

    QString fichier_ouvrir = "";
    if ((debut_str_plan != 0) && (fin_str_plan != 0))
    {
        for(int i=debut_str_plan; i<=fin_str_plan; ++i)
        {
            fichier_ouvrir += arguments[i];
            if (i < (fin_str_plan))
                fichier_ouvrir += " "; /* Restaure l'espace */
        }
    }

    if (fichier_ouvrir.isEmpty()) /* Si rien n'a été trouvé en fichier à ouvrir via un chemin, on tente avec le nom direct: */
    {
        for(unsigned int i=1; i<taille_args; ++i)
        {
            if (arguments[i].contains(".dxf", Qt::CaseInsensitive) || arguments[i].contains("." ExtensionScriptsAutomate, Qt::CaseInsensitive))
            {
                fichier_ouvrir = arguments[i];
                break;
            }
        }
    }

    const int id_style = (style ? 1 : 0);

    if (!assigne_style)
    {
#if defined(PSYS_LINUX)

        /* Recherche le style breeze */
        bool trouve_style = false;
        const QStringList liste_styles = QStyleFactory::keys();

#ifdef DEBUG
//        qDebug() << "Liste des styles disponibles pour l'interface : " << liste_styles;
#endif

        for(int i=0; i<liste_styles.size(); ++i)
        {
            const QString &s = liste_styles[i];
            if (s.compare("breeze", Qt::CaseInsensitive) == 0)
            {
                QApplication::setStyle(QStyleFactory::create("breeze"));
                trouve_style = true;
            }
        }

        if (!trouve_style)
        {
            QApplication::setStyle(QStyleFactory::create("fusion"));
        }


#elif defined(PSYS_WINDOWS)
        QApplication::setStyle(QStyleFactory::create("windows"));
#endif
    }

//#ifndef DEBUG
    if (style)
    {
        /* Assigne la palette (en cas de manquement dans les feuilles de style) */
        QPalette palette;
        const QColor couleur_fond(49, 54, 59);
        const QColor couleur_base(couleur_fond);
        const QColor couleur_texte(220, 220, 220);
        const QColor couleur_texte_surligne(61, 174, 233);
        const QColor couleur_surligne(couleur_fond);

        palette.setColor(QPalette::Window, couleur_fond);
        palette.setColor(QPalette::WindowText, couleur_texte);
        palette.setColor(QPalette::Base, couleur_base);
        palette.setColor(QPalette::AlternateBase, couleur_fond);
        palette.setColor(QPalette::ToolTipBase, couleur_base);
        palette.setColor(QPalette::ToolTipText, couleur_texte);
        palette.setColor(QPalette::Text, couleur_texte);
        palette.setColor(QPalette::Button, couleur_fond);
        palette.setColor(QPalette::ButtonText, couleur_texte);
        palette.setColor(QPalette::BrightText, couleur_texte_surligne);
        palette.setColor(QPalette::Highlight, couleur_surligne.lighter());
        palette.setColor(QPalette::HighlightedText, Qt::black);
        app.setPalette(palette);

        ChargeStyleApp(app, id_style);
    }
//#endif

//    qRegisterMetaType<bool>("bool");
//    qRegisterMetaType<int>("int");
//    qRegisterMetaType<float>("float");
//    qRegisterMetaType<double>("double");
    qRegisterMetaType<QRgb>("QRgb");
    qRegisterMetaType<Perspective3D::vues2D_t>("Perspective3D::vues2D_t");
//    qRegisterMetaType<Perspective3D::PCouleur>("Perspective3D::PCouleur");
    qRegisterMetaType<Perspective3D::ParametresPerspective>("Perspective3D::ParametresPerspective");
#ifdef SUPPORT_VISION2D
    qRegisterMetaType<Perspective3D::ParametresVision>("Perspective3D::ParametresVision");
#endif // SUPPORT_VISION
    MainWindow w(lang_p3d, chemin_zetta6, gen_auto, fichier_ouvrir, id_style);

#ifdef Q_OS_WIN
//    QWindowsWindowFunctions::setHasBorderInFullScreen(w.window(), true); /* https://bugreports.qt.io/browse/QTBUG-7556 */
#endif // Q_OS_WIN

    w.show();

    return app.exec();
}
