﻿/**
© Florian Joncour, 2013-2018 florian@zetta-sys.com

Ce logiciel est un programme informatique faisant interface à la bibliothèque
Perspective3D, un outil de modélisation 3D à partir de vues orthographiques 2D.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site "http://www.cecill.info".

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée.  Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme,  le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard  l'attention de l'utilisateur est attirée sur les risques
associés au chargement,  à l'utilisation,  à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant  des  connaissances  informatiques approfondies.  Les
utilisateurs sont donc invités à charger  et  tester  l'adéquation  du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.
**/

#include <QHelpSearchEngine>
#include <QCloseEvent>
#include <QDir>

#include <QWidget>
#include <QSplitter>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QLabel>
#include <QFrame>
#include <QSpacerItem>
//#include <QApplication>

#if QT_VERSION >= QT_VERSION_CHECK(5,0,0)
#include <QHelpContentWidget>
#else
#include <QtHelp/QHelpContentWidget>
#endif

#include <QDesktopServices>

#include "animeQt.h"
#include "documentation.h"

#ifdef DEBUG
#include <QDebug>
#endif

/* Configuration des langues disponibles pour la documentation: */
const char * const Str_Langues[] = { "fr", "\0" };
const unsigned int N_Langues = 1;

NavigateurDoc::NavigateurDoc(QHelpEngine *moteur, const QString &chemin_bin_app, QWidget *parent) : QTextBrowser(parent), moteurAide(moteur)
{
    QDir chemin(chemin_bin_app);
    chemin.cdUp(); /* Le chemin de la racine de l'application est parente du répertoire du binaire. */
    chemin_racine_app = chemin.absolutePath();
    setOpenExternalLinks(false);
}

QVariant NavigateurDoc::loadResource(int type, const QUrl &url)
{
    const QString protocole = url.scheme();
    if (protocole == "qthelp")
    {
        return QVariant(moteurAide->fileData(url));
    }
    else if (protocole == "exec")
    {
        QString copie_str(url.toString());
        copie_str.replace("exec://", "");

        if (QDir::isAbsolutePath(copie_str))
        {

            if (QFile::exists(copie_str))
            {
                ;
            }
            else
            {
//                copie_str.clear();
            }
        }
        else /* Chemin relatif. */
        {
            copie_str.prepend(QString("scripts") + QDir::separator());
            if (QFile::exists(copie_str))
            {
                ;
            }
            else
            {
                copie_str = chemin_racine_app + QDir::separator() + copie_str;
                if (!QFile::exists(copie_str))
                {
//                    copie_str.clear();
                }
            }
        }

        QFile fichier(copie_str);
//        QApplication::setOverrideCursor(Qt::ArrowCursor);  /* Sinon ça reste bloqué sur Qt::WaitCursor */

        emit ExecFichier(copie_str);
        return QVariant(QString("<html><body><div><a href=\"") + source().toString() + "\"\\>" + tr("Annuler") + QString("</a><br /></div><div style=\"margin-top:100px\"><center><h1>") + tr("Ouverture du fichier :<br />%1").arg(copie_str) + QString("</h1></div></center><br /><div>" + ((!copie_str.isEmpty() && fichier.exists()) ? fichier.readAll().replace("\n", "<br />") : tr("Fichier introuvable.")) + "</div></body></html>"));
    }
    else if (protocole == "conf")
    {
        QString copie_str(url.toString());
        copie_str.replace("conf://", "");
        emit OuvreConfig(copie_str);
        return QVariant(QString("<html><body><div><a href=\"") + source().toString() + "\"\\>" + tr("Retour") + QString("</a><br /></div></body></html>"));
    }
    else if (protocole == "http" || protocole == "https") /* Lien web */
    {
        QDesktopServices::openUrl(url);
        return QVariant(QString("<html><body><div><a href=\"") + source().toString() + "\"\\>" + tr("Annuler") + QString("</a><br /><div style=\"margin-top:100px\"><center><h1>") + tr("Redirection vers le navigateur :<br />%1").arg(url.toString()) + QString("</h1></div></center></body></html>"));
    }
    return QTextBrowser::loadResource(type, url);
}

void Documentation::OuvrePage(const QString &page, bool force_affichage)
/* Ouvre une page en particulier dans la documentation. */
{
    if (!ContenuAide)
        return;
    if (!isVisible() && force_affichage)
        show();
    ContenuAide->setSource(QUrl(QString("qthelp://zetta-sys.com/doc/") + page));
//    ContenuAide->document()->adjustSize();
}

QString Documentation::PageCourante() const
{
    if (!ContenuAide)
        return QString("");
    QUrl url = ContenuAide->source();

#if QT_VERSION >= QT_VERSION_CHECK(5,0,0)
    QString str = url.fileName();
#else
    QString str = url.toString().section("/",-1,-1);
#endif

    if (url.hasFragment())
        str += "#" + url.fragment();
    return str;
}

QString Documentation::CheminDocumentation(const QString &chemin_zetta6) const
{
    bool valide_chemin_doc = false;
    QString chemin_doc = chemin_zetta6 + "/doc/" + Str_Langues[Langue]+".qhc";
    if (QFile::exists(chemin_doc))
    {
        valide_chemin_doc = true;
    }
    else /* Tente le répertoire parent (suivant la manière dont a été lancé Zetta6, la documentation peut être dans un sous dossier par rapport au binaire). */
    {
        QDir chemin_installation_p(chemin_zetta6);
        chemin_installation_p.cdUp();
        chemin_doc = chemin_installation_p.absolutePath() + "/doc/" + Str_Langues[Langue]+".qhc";

        valide_chemin_doc = QFile::exists(chemin_doc);
    }

    if (valide_chemin_doc)
    {
        return chemin_doc;
    }
    return QString("");
}

//void Documentation::closeEvent(QCloseEvent *ev)
//{
//    ev->accept();
//}

Documentation::Documentation(const QString &chemin_zetta6, unsigned int langue, QWidget *parent) : QDockWidget(tr("Documentation"), parent)
{
    if (langue >= N_Langues)
        langue = 0; // Langue par défaut si mauvais id donné.

    Langue = langue;

    QString chemin_documentation = CheminDocumentation(chemin_zetta6);

    WidgetContenu = new QWidget(this);
    LayoutContenu = new QVBoxLayout;
    WidgetContenu->setLayout(LayoutContenu);

    Aide = new QHelpEngine(chemin_documentation, this);
    Aide->setupData();
    setObjectName(tr("AideD"));
//    setAllowedAreas(Qt::AllDockWidgetAreas);
    setAllowedAreas(Qt::RightDockWidgetArea);
    ContenuAide = new NavigateurDoc(Aide, chemin_zetta6, this);

    LayoutNavigation = new QHBoxLayout();
    LayoutNavigation->setObjectName("LayoutNavigation");
    LayoutNavigation->setSpacing(6);
    LayoutNavigation->setContentsMargins(16, 6, 16, 6);
    LayoutContenu->addLayout(LayoutNavigation);

    BoutonPrecedent = new QPushButton();
    BoutonPrecedent->setObjectName("aide"); /* Pour le style *petit bouton* */
    BoutonPrecedent->setFixedHeight(24);
    BoutonPrecedent->setFixedWidth(22);
    BoutonPrecedent->setIconSize(QSize(10, 10));
    BoutonPrecedent->setIcon(QIcon(":/icones/fleche_gauche.png"));
    BoutonPrecedent->setText("");
    BoutonPrecedent->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    LayoutNavigation->addWidget(BoutonPrecedent);

    BoutonSuivant = new QPushButton();
    BoutonSuivant->setObjectName("aide"); /* Pour le style *petit bouton* */
    BoutonSuivant->setFixedHeight(24);
    BoutonSuivant->setFixedWidth(22);
    BoutonSuivant->setIconSize(QSize(10, 10));
    BoutonSuivant->setIcon(QIcon(":/icones/fleche_droite.png"));
    BoutonSuivant->setText("");
    BoutonSuivant->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    LayoutNavigation->addWidget(BoutonSuivant);

    LabelNavigation = new QLabel();
    LabelNavigation->setFrameStyle(QFrame::StyledPanel | QFrame::Plain);
    LabelNavigation->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Minimum);
    LabelNavigation->setContentsMargins(3,0,3,0);
    LayoutNavigation->addWidget(LabelNavigation);

    LayoutNavigation->addSpacing(16);

    BoutonAccueil = new QPushButton();
    BoutonAccueil->setObjectName("aide"); /* Pour le style *petit bouton* */
    BoutonAccueil->setFixedHeight(24);
    BoutonAccueil->setFixedWidth(22);
    BoutonAccueil->setIconSize(QSize(10, 10));
    BoutonAccueil->setIcon(QIcon(":/icones/retour.png"));
    BoutonAccueil->setText("");
    BoutonAccueil->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    LayoutNavigation->addWidget(BoutonAccueil);

    EspacementNavigation = new QSpacerItem(40, 26, QSizePolicy::MinimumExpanding, QSizePolicy::Minimum);
    LayoutNavigation->addItem(EspacementNavigation);

    Separateur = new QSplitter(Qt::Horizontal, this);

    Aide->contentWidget()->setMaximumWidth(250);
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
//    setMinimumSize(QSize(640, 480));
//    setMinimumSize(QSize(750, 560));

    Aide->contentWidget()->setMinimumWidth(140);
    Aide->contentWidget()->setMaximumWidth(140);
    Aide->contentWidget()->hide(); /* Masque la barre latérale de navigation (elle bouffe trop de place). */
    Aide->contentWidget()->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
    ContenuAide->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    ContenuAide->setContextMenuPolicy(Qt::NoContextMenu);
    ContenuAide->setMinimumWidth(530); // 830
    ContenuAide->setMinimumHeight(500);

    LayoutContenu->addWidget(ContenuAide);

    Separateur->insertWidget(0, Aide->contentWidget());
    Separateur->insertWidget(1, WidgetContenu);

    setWidget(Separateur);
    connect(Aide->contentWidget(), SIGNAL(linkActivated(const QUrl &)), ContenuAide, SLOT(setSource(const QUrl &)));
    connect(ContenuAide, SIGNAL(ExecFichier(const QString &)), this, SLOT(ExecFichier_s(const QString &)));
    connect(ContenuAide, SIGNAL(OuvreConfig(const QString &)), this, SLOT(OuvreConfig_s(const QString &)));
    connect(ContenuAide, SIGNAL(sourceChanged(const QUrl &)), this, SLOT(sourceChanged(const QUrl &)));
    connect(BoutonPrecedent, SIGNAL(pressed()), this, SLOT(PrecedentHistorique()));
    connect(BoutonSuivant, SIGNAL(pressed()), this, SLOT(SuivantHistorique()));
    connect(BoutonAccueil, SIGNAL(pressed()), this, SLOT(Accueil()));

    Index();

//    qDebug() << "Erreurs navigateur aide: " << Aide->error();
}

Documentation::~Documentation()
{
    delete BoutonPrecedent;
    delete BoutonSuivant;
    delete BoutonAccueil;
    delete LabelNavigation;
//    delete EspacementNavigation;
    delete LayoutNavigation;
    delete LayoutContenu;
    delete ContenuAide;
    delete WidgetContenu;
    delete Aide;
    delete Separateur;
}

void Documentation::sourceChanged(const QUrl &s)
{
    BoutonPrecedent->setEnabled(ContenuAide->isBackwardAvailable());
    BoutonSuivant->setEnabled(ContenuAide->isForwardAvailable());
    LabelNavigation->setText(s.toString().replace("qthelp://zetta-sys.com/", "").replace(".html", ""));
}

void Documentation::Index()
{
    OuvrePage("index.html");
}

void Documentation::PrecedentHistorique()
{
    if (ContenuAide->isBackwardAvailable())
    {
        ContenuAide->backward();
    }
}

void Documentation::SuivantHistorique()
{
    if (ContenuAide->isForwardAvailable())
    {
        ContenuAide->forward();
    }
}

void Documentation::Accueil()
{
    Index();
}

void Documentation::ExecFichier_s(const QString &s)
{
    emit ExecFichier(s);
}

void Documentation::OuvreConfig_s(const QString &s)
{
    emit OuvreConfig(s);
}

BoutonAide::BoutonAide(Documentation *doc, const char *url, QWidget *parent) : QPushButton(parent)
{
    lien = QString(url);
    documentation = doc;
    setText(tr("?"));
    setMinimumSize(QSize(25,25));
    setMaximumSize(QSize(25,25));
    connect(this, SIGNAL(pressed()), this, SLOT(click()));
}

void BoutonAide::click()
{
    documentation->OuvrePage(lien);
}

BoutonAide::~BoutonAide()
{
    ;
}
