﻿/**
© Florian Joncour, 2013-2018 florian@zetta-sys.com

Ce logiciel est un programme informatique faisant interface à la bibliothèque
Perspective3D, un outil de modélisation 3D à partir de vues orthographiques 2D.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site "http://www.cecill.info".

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée.  Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme,  le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard  l'attention de l'utilisateur est attirée sur les risques
associés au chargement,  à l'utilisation,  à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant  des  connaissances  informatiques approfondies.  Les
utilisateurs sont donc invités à charger  et  tester  l'adéquation  du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.
**/

#include <iostream>
#include <QFile>

#include "zcalc.h"
#include "eval_exp.h"

/* Extension du langage (ajout de fonctions personnalisées): */
Evaluateur::def_fonction Fonctions_Extension[] = { { "message", Evaluateur::FONCTION, -1, 0 } };
const unsigned int n_Fonctions_Extension = sizeof(Fonctions_Extension) / sizeof(Evaluateur::def_fonction);

class DLL_LOCAL ExtensionLangage : public Evaluateur::DictionnaireLangage
{
    public:
        inline explicit ExtensionLangage(Zcalc *parent=0) : Parent(parent)
        {
        }

        inline ~ExtensionLangage()
        {
        }

        inline bool Exec(unsigned int id_fonction, Evaluateur::operation_tok *args, unsigned int n_args) const
        {
            const unsigned int domaine_fonction = DOMAINE_FCT(id_fonction);
            const unsigned int fonction = ID_FCT(id_fonction);

            if (domaine_fonction != Evaluateur::DOM_FONCTION_EXTERNE)
            {
                return false;
            }

            if (fonction == 0)
            {
                if (n_args == 0)
                {
                    return false;
                }
                fct_message(args, n_args);
                return true;
            }
            else
            {
                return false;
            }
            return false;
        }

        inline int IdFonction(const std::string &s) const
        {
            return IdFonctionDom(s, Evaluateur::DOM_FONCTION_EXTERNE);
        }

        inline int IdFonctionDom(const std::string &s, unsigned int domaine) const
        {
            if (domaine == Evaluateur::DOM_FONCTION_EXTERNE)
            {
                for (unsigned int i=0; i<n_Fonctions_Extension; ++i)
                {
                    if (Evaluateur::CompareStrEval(s, Fonctions_Extension[i].nom_fonction))
                    {
                        return i;
                    }
                }
            }
            return -1;
        }

        inline const Evaluateur::def_fonction &ReferenceFonction(unsigned int id) const
        {
            unsigned int id_domaine = DOMAINE_FCT(id);
            unsigned int id_fct = ID_FCT(id);

            if (id_domaine == Evaluateur::DOM_FONCTION_EXTERNE)
            {
                if (id_fct < n_Fonctions_Extension)
                {
                    return Fonctions_Extension[id_fct];
                }
            }
            return Evaluateur::FonctionNulle;
        }

        inline bool fct_message(Evaluateur::operation_tok *args, const unsigned int n_args) const
        {
            if (Parent)
            {
                QString mesg;
                for(unsigned int i=0; i<n_args; ++i)
                {
                    if (i)
                    {
                        mesg += ", ";
                    }
                    mesg += QString::fromStdString(args[i].resultat.encode());
                }
                Parent->Message(mesg);
                return true;
            }
            return false;
        }

    private:
        Zcalc *Parent;
};

Zcalc::Zcalc(QObject *parent) : QObject(parent)
{
    extension = new ExtensionLangage(this);
    evaluateur = new Evaluateur::EvaluateurExp(extension);
//    evaluateur = new Evaluateur::EvaluateurExp(0);
}

Zcalc::~Zcalc()
{
    delete evaluateur;
    delete extension;
}

void Zcalc::Reinit()
{
    evaluateur->Reinit();
}

bool Zcalc::ValideResultat()
{
    return evaluateur->EtatMachine().valide;
}

bool Zcalc::AjoutVariableBool(const QString &var, bool n)
{
    return (evaluateur->AjoutVariableBool(var.toStdString(), n));
}

bool Zcalc::AjoutVariableInt(const QString &var, int n)
{
    return (evaluateur->AjoutVariableInt(var.toStdString(), Evaluateur::eval_int(n)));
}

bool Zcalc::AjoutVariableFloat(const QString &var, double n)
{
    return (evaluateur->AjoutVariableFloat(var.toStdString(), Evaluateur::eval_float(n)));
}

bool Zcalc::ValeurFloat(const QString &var, double &n) const
/* Assigne la valeur de la variable donnée en argument sous forme décimale et renvoi true si la conversion est possible, sinon false. */
{
    return (evaluateur->ValeurFloat(var.toStdString(), true, n));
}

bool Zcalc::ValeurInt(const QString &var, int &n) const
/* Assigne la valeur de la variable donnée en argument sous forme entière et renvoi true si la conversion est possible, sinon false. */
{
    return (evaluateur->ValeurInt(var.toStdString(), true, n));
}

bool Zcalc::ValeurShort(const QString &var, short &n) const
/* Assigne la valeur de la variable donnée en argument sous forme entière et renvoi true si la conversion est possible, sinon false. */
{
    return (evaluateur->ValeurShort(var.toStdString(), true, n));
}

bool Zcalc::ValeurBool(const QString &var, bool &n) const
/* Assigne la valeur de la variable donnée en argument sous forme booléenne et renvoi true si la conversion est possible, sinon false. */
{
    return (evaluateur->ValeurBool(var.toStdString(), true, n));
}

QString Zcalc::EncodeValeur(const QString &var) const
/* Envoi la valeur encodée sous forme de chaîne de caractères de la variable donnée en argument. */
{
    const Evaluateur::variable_tok &v = evaluateur->VariableConst(var.toStdString(), true, true);
    if (v.valeur.valide)
    {
        return QString::fromStdString(v.valeur.encode());
    }
    return QString();
}

void Zcalc::Message(const QString &s)
{
    emit Message_sig(s);
}

bool Zcalc::ExecCode(const QString &code)
{
    return evaluateur->Exec(code.toStdString());
}

bool Zcalc::ExecCond(const QString &code, bool var_sortie)
{
    return evaluateur->ExecCond(code.toStdString(), var_sortie);
}

bool Zcalc::ExecFichier(const QString &chemin_script)
{
    if (QFile::exists(chemin_script))
    {
        QFile f_scr(chemin_script);
        if (f_scr.open(QFile::ReadOnly | QFile::Text))
        {
            QString txt = QString::fromUtf8(f_scr.readAll());
            f_scr.close();
            return (this->ExecCode(txt));
        }
    }
    return false;
}
