﻿/**
© Florian Joncour, 2013-2018 florian@zetta-sys.com

Ce logiciel est un programme informatique faisant interface à la bibliothèque
Perspective3D, un outil de modélisation 3D à partir de vues orthographiques 2D.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site "http://www.cecill.info".

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée.  Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme,  le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard  l'attention de l'utilisateur est attirée sur les risques
associés au chargement,  à l'utilisation,  à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant  des  connaissances  informatiques approfondies.  Les
utilisateurs sont donc invités à charger  et  tester  l'adéquation  du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.
**/

#include <QPushButton>
#include <QDialogButtonBox>

#include "dialogactivation.h"
#include "API/perspective_api.h"

DialogActivation::DialogActivation(puint64 salage_licence, QWidget *parent) : QDialog(parent), SalageLicence(salage_licence)
{
    ui.setupUi(this);

#ifdef DEBUG
    ui.buttonBox->setStandardButtons(QDialogButtonBox::Ok | QDialogButtonBox::Close);
    ui.buttonBox->button(QDialogButtonBox::Ok)->setText(tr("Ok"));
#endif

    ui.buttonBox->button(QDialogButtonBox::Close)->setText(tr("Fermer"));

    ui.led_cle->clear();
    ui.led_cle->setInputMask(">NNNNNN-NNNNNN-NNNNNN-NNNNNN;#");
//    ui.led_cle->setEchoMode(QLineEdit::Password);

    setWindowModality(Qt::WindowModal);

    for(unsigned int i=0; i<TailleCleActivation_Sep; ++i)
    {
        Cle[i] = '\0';
    }

    valide = false;
    exec();
}

const QString &DialogActivation::StyleContourLicence(const bool activ_2d, const bool activ_3d)
{
    static const QString           activ_nul("QLineEdit, QLabel { border: 2px solid red;   border-radius: 0px; }");
    static const QString       activ_complet("QLineEdit, QLabel { border: 2px solid green; border-radius: 0px; }");
    static const QString activ_3d_uniquement("QLineEdit, QLabel { border: 2px solid blue;  border-radius: 0px; }");

#ifdef SUPPORT_VISION2D
    static const QString activ_2d_uniquement("border: 2px solid grey;  border-radius: 0px;");

    if (activ_2d || activ_3d)
    {
        if (activ_2d && activ_3d)
        {
            return activ_complet;
        }
        else if (activ_2d)
        {
            return activ_2d_uniquement;
        }
        else if (activ_3d)
        {
            return activ_3d_uniquement;
        }
    }
#else
    if (activ_3d)
    {
        return activ_complet;
    }
#endif
    return activ_nul;
}

void DialogActivation::GenCleActiv(const QString &txt)
/* Génère la clé d'activation chiffrée */
{
    pint8 tampon_source[TailleCleActivation_Sep];

    QByteArray data_source = txt.toLocal8Bit();
    int i;
    for(i=0; i<txt.size(); ++i)
    {
        if (i>=TailleCleActivation_Sep)
        {
            i = 0;
            break;
        }
        tampon_source[i] = data_source[i];
    }
    if (i < (TailleCleActivation_Sep-1))
    {
        for(unsigned int i2=i; i<TailleCleActivation_Sep; ++i)
        {
            tampon_source[i2] = '\0';
        }
    }
    libPerspective::ChiffrementLicence(tampon_source, Cle, SalageLicence);
}

const pint8 *DialogActivation::CleCR() const
/* Renvoi la clé chiffrée. */
{
    if (valide)
    {
        return Cle;
    }
    return nullptr;
}

void DialogActivation::timerEvent(QTimerEvent *ev)
{
    if (valide)
    {
        close();
    }
    ev->accept();
}

void DialogActivation::on_led_cle_textEdited(const QString &arg1)
{
    GenCleActiv(arg1);

    const bool v_3d = Perspective3D::Perspective::ValideLicence(Cle, SalageLicence);

#ifdef SUPPORT_VISION2D
    const bool v_2d = Perspective3D::Vision::ValideLicence(Cle, SalageLicence);
#else
    const bool v_2d = false;
#endif // SUPPORT_VISION2D

    if (v_2d || v_3d) // Activation valide
    {
        ui.lbl_activ->setPixmap(QPixmap(":/icones/valide.png"));
//        ui.led_cle->setStyleSheet("border: 2px solid green; border-radius: 0px;");
        ui.led_cle->setStyleSheet(DialogActivation::StyleContourLicence(v_2d, v_3d));
#ifndef DEBUG
        valide = true;
        startTimer(450); // Délai avant fermeture automatique de la fenêtre
#endif // DEBUG
    }
    else
    {
        ui.lbl_activ->setPixmap(QPixmap(":/icones/invalide.png"));
//        ui.led_cle->setStyleSheet("border: 2px solid red; border-radius: 0px;");
        ui.led_cle->setStyleSheet(DialogActivation::StyleContourLicence(v_2d, v_3d));
#ifndef DEBUG
        valide = false;
#endif // DEBUG
    }
}

void DialogActivation::on_buttonBox_clicked(QAbstractButton *button)
{
#ifdef DEBUG
    if (ui.buttonBox->buttonRole(button) == QDialogButtonBox::AcceptRole)
    {
        valide = true;
        close();
    }
    else if (ui.buttonBox->buttonRole(button) == QDialogButtonBox::RejectRole)
    {
        valide = false;
        close();
    }
#else
    if (ui.buttonBox->buttonRole(button) == QDialogButtonBox::RejectRole)
    {
        close();
    }
#endif // DEBUG
}
