﻿#include "dialogfichiers.h"

#include "automate.h"
#include <QLabel>
#include <QGridLayout>
#include <QFileIconProvider>

#define TAILLE_IMAGES 128

TypesFichiers TypeFichier(const QString &s)
{
    if (s.endsWith(".dxf", Qt::CaseInsensitive))
        return TypesFichiers::DXF_3D;
    if (s.endsWith(".bmp", Qt::CaseInsensitive))
        return TypesFichiers::BMP;
    if (s.endsWith(".png", Qt::CaseInsensitive))
        return TypesFichiers::PNG;
    if (s.endsWith(".jpg", Qt::CaseInsensitive) || s.endsWith(".jpeg", Qt::CaseInsensitive))
        return TypesFichiers::JPG;
    if (s.endsWith(".tga", Qt::CaseInsensitive))
        return TypesFichiers::TGA;
    if (s.endsWith(".gif", Qt::CaseInsensitive))
        return TypesFichiers::GIF;
    if (s.endsWith(QString("." ExtensionScriptsAutomate), Qt::CaseInsensitive))
        return TypesFichiers::Z6S;
    if (s.endsWith(".stl", Qt::CaseInsensitive))
        return TypesFichiers::STL;
    if (s.endsWith(".3ds", Qt::CaseInsensitive))
        return TypesFichiers::x3DS;
    if (s.endsWith(".obj", Qt::CaseInsensitive))
        return TypesFichiers::OBJ;
    if (s.endsWith(".x3d", Qt::CaseInsensitive))
        return TypesFichiers::X3D;
    if (s.endsWith(".dae", Qt::CaseInsensitive))
        return TypesFichiers::DAE;
    if (s.endsWith(".ply", Qt::CaseInsensitive))
        return TypesFichiers::PLY;
    if (s.endsWith(".amf", Qt::CaseInsensitive))
        return TypesFichiers::AMF;

    QFileInfo info(s);
    if (info.isDir())
    {
        return TypesFichiers::DOSSIER;
    }
    return TypesFichiers::NUL; /* Inconnu */
}

class ProviderIconesDialog : public QFileIconProvider
{
    public:
        inline ProviderIconesDialog() : QFileIconProvider()
        {
            ;
        }

        inline QIcon icon(const QFileInfo &info) const
        {
            const TypesFichiers type_fichier = TypeFichier(info.fileName());
            if (type_fichier == TypesFichiers::BMP || type_fichier == TypesFichiers::PNG || type_fichier == TypesFichiers::JPG || type_fichier == TypesFichiers::TGA || type_fichier == TypesFichiers::GIF)
            {
                return QIcon(":/icones/plan_image.png"); /* Version rapide... */

//                QPixmap img(info.filePath());
////                return QIcon(img.scaled(64, 64, Qt::KeepAspectRatio, Qt::FastTransformation));
//                return QIcon(img.scaled(64, 64, Qt::KeepAspectRatioByExpanding, Qt::SmoothTransformation));
            }
            else if (type_fichier == TypesFichiers::DXF_3D)
            {
                return QIcon(":/icones/plan_dxf.png");
            }
            else if (type_fichier == TypesFichiers::Z6S)
            {
                return QIcon(":/icones/script.png");
            }
            else if ( type_fichier == TypesFichiers::STL  ||
                           type_fichier == TypesFichiers::x3DS ||
                           type_fichier == TypesFichiers::OBJ  ||
                           type_fichier == TypesFichiers::X3D  ||
                           type_fichier == TypesFichiers::DAE  ||
                           type_fichier == TypesFichiers::PLY  ||
                           type_fichier == TypesFichiers::AMF  )
            {
                return QIcon(":/icones/cube_perspective.png");
            }
//            else if (type_fichier == TypesFichiers::DOSSIER)
//            {
//                return QIcon(":/icones/parcourir.png");
//            }
            return QFileIconProvider::icon(info);
        }
};

QFileDialogP::QFileDialogP(QWidget* parent, const QString& caption, const QString& directory, const QString& filter) :
    QFileDialog(parent, caption, directory, filter)
{
    setOption(QFileDialog::DontUseNativeDialog, true);
    provider_icones_dialog = new ProviderIconesDialog;
    setIconProvider(provider_icones_dialog);

    setObjectName("QFileDialogP");
    setMinimumSize(800+TAILLE_IMAGES, 600);

    QGridLayout *layout = qobject_cast<QGridLayout *>(this->layout());

    if (layout != nullptr)
    {
        QVBoxLayout* box = new QVBoxLayout;

        entete = new QLabel("", this);
        entete->setAlignment(Qt::AlignCenter);
        entete->setObjectName("QFileDialogP");
        entete->setMinimumWidth(TAILLE_IMAGES);
        entete->hide();

        img_apercu = new QLabel("", this);
        img_apercu->setAlignment(Qt::AlignCenter);
        img_apercu->setObjectName("QFileDialogP");
        img_apercu->setMinimumSize(TAILLE_IMAGES, TAILLE_IMAGES);
        img_apercu->hide();

        informations = new QPlainTextEdit("", this);
        informations->setObjectName("QFileDialogP");
        informations->setMinimumWidth(TAILLE_IMAGES);
        informations->setMaximumWidth(TAILLE_IMAGES);
        informations->setMinimumHeight(600-(TAILLE_IMAGES+132)); // 112
        informations->setReadOnly(true);
        informations->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
        informations->setHorizontalScrollBarPolicy(Qt::ScrollBarAsNeeded);
        informations->setLineWrapMode(QPlainTextEdit::NoWrap);
        informations->hide();

        box->addWidget(entete);
        box->addSpacing(15);
        box->addWidget(img_apercu);
        box->addSpacing(15);
        box->addWidget(informations);
        box->addStretch();

        QList< QPair<QLayoutItem*, QList<int> > > movedItems;
        for(int i = 0; i < layout->count(); i++)
        {
            int row, column, rowSpan, columnSpan;
            layout->getItemPosition(i,&row,&column,&rowSpan,&columnSpan);
            if (row > 2)
            {
                QList<int> list;
                list << row << column << rowSpan << columnSpan;
                movedItems << qMakePair(layout->takeAt(i), list);
                --i;
            }
        }
        for(int i = 0; i < movedItems.count(); i++)
        {
            layout->addItem(movedItems[i].first, movedItems[i].second[0], movedItems[i].second[1], movedItems[i].second[2], movedItems[i].second[3]);
        }

        layout->addItem(box,1,3,1,1);
        connect(this, SIGNAL(currentChanged(const QString&)), this, SLOT(OnCurrentChanged(const QString&)));
    }
    else
    {
        entete = img_apercu = nullptr;
        informations = nullptr;
#ifdef DEBUG
        Q_ASSERT(false);
#endif
    }
}

QFileDialogP::~QFileDialogP()
{
    delete provider_icones_dialog;
}

void QFileDialogP::OnCurrentChanged(const QString &chemin)
{
    if (entete == nullptr || img_apercu == nullptr || informations == nullptr)
    {
        return;
    }

    const TypesFichiers type_fichier = TypeFichier(chemin);

    if (type_fichier == TypesFichiers::BMP || type_fichier == TypesFichiers::PNG || type_fichier == TypesFichiers::JPG || type_fichier == TypesFichiers::TGA || type_fichier == TypesFichiers::GIF)
    {
        QPixmap pixmap = QPixmap(chemin);
        entete->setText(tr("Image"));
        entete->show();
        img_apercu->setMinimumSize(TAILLE_IMAGES, TAILLE_IMAGES);
        img_apercu->setPixmap(pixmap.scaled(img_apercu->width(), img_apercu->height(), Qt::KeepAspectRatio, Qt::SmoothTransformation));
        img_apercu->show();
        informations->clear();
        informations->hide();
    }
    else if (type_fichier == TypesFichiers::DXF_3D)
    {
        QPixmap pixmap(":/icones/plan_dxf_128.png");
        entete->setText(tr("Plan DXF"));
        entete->show();
        img_apercu->setMinimumSize(pixmap.width(), pixmap.height());
        img_apercu->setPixmap(pixmap);
        img_apercu->show();
        informations->clear();
        informations->hide();
    }
    else if (type_fichier == TypesFichiers::Z6S)
    {
        QPixmap pixmap(":/icones/script_128.png");
        entete->setText(tr("Script Zetta6"));
        entete->show();
        img_apercu->setMinimumSize(pixmap.width(), pixmap.height());
        img_apercu->setPixmap(pixmap);
        img_apercu->show();
        informations->show();

        QFile f_scr(chemin);
        if (f_scr.open(QFile::ReadOnly | QFile::Text))
        {
            QString contenu_script;
            QString txt = QString::fromUtf8(f_scr.readAll());
            f_scr.close();
            informations->setPlainText(txt);

//            QStringList ls_txt = txt.split(QRegExp("[\r\n]"), QString::KeepEmptyParts);

//            const int n_lignes = ls_txt.size() <= 20 ? ls_txt.size() : 20;
//            for(int i=0; i<n_lignes; ++i)
//            {
//                if (ls_txt[i].length() > 15)
//                {
//                    ls_txt[i].resize(15);
//                    contenu_script += "\n" + ls_txt[i] + "...";
//                }
//                else
//                {
//                    contenu_script += "\n" + ls_txt[i];
//                }
//            }

//            if (contenu_script.size())
//            {
//                if (n_lignes > ls_txt.size())
//                {
//                    contenu_script += "...";
//                }
//                informations->setPlainText(contenu_script);
//            }
        }
    }
    else if ( type_fichier == TypesFichiers::STL  ||
              type_fichier == TypesFichiers::x3DS ||
              type_fichier == TypesFichiers::OBJ  ||
              type_fichier == TypesFichiers::X3D  ||
              type_fichier == TypesFichiers::DAE  ||
              type_fichier == TypesFichiers::PLY  ||
              type_fichier == TypesFichiers::AMF  )
    {
        QFileInfo infos_fichier(chemin);

        QPixmap pixmap(":/icones/cube_perspective_128.png");
//        QPixmap pixmap = provider_icones_dialog->icon(infos_fichier).pixmap(128, 128);

        entete->setText(tr("Solide") + " " + infos_fichier.completeSuffix().toUpper());
        entete->show();
        img_apercu->setMinimumSize(pixmap.width(), pixmap.height());
        img_apercu->setPixmap(pixmap);
        img_apercu->show();
        informations->clear();
        informations->hide();
    }
//    else if (type_fichier == TypesFichiers::DOSSIER)
//    {
//        QPixmap pixmap(":/icones/parcourir.png");
//        entete->setText(tr("Dossier"));
//        entete->show();
//        img_apercu->setMinimumSize(pixmap.width(), pixmap.height());
//        img_apercu->setPixmap(pixmap);
//        img_apercu->show();
//        informations->clear();
//        informations->hide();
//    }
    else /* Aucune miniature */
    {
        entete->clear();
        entete->hide();
        img_apercu->clear();
        img_apercu->setMinimumSize(0, 0);
        img_apercu->hide();
        informations->clear();
        informations->hide();
    }
}
