﻿/**
© Florian Joncour, 2013-2018 florian@zetta-sys.com

Ce logiciel est un programme informatique faisant interface à la bibliothèque
Perspective3D, un outil de modélisation 3D à partir de vues orthographiques 2D.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site "http://www.cecill.info".

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée.  Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme,  le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard  l'attention de l'utilisateur est attirée sur les risques
associés au chargement,  à l'utilisation,  à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant  des  connaissances  informatiques approfondies.  Les
utilisateurs sont donc invités à charger  et  tester  l'adéquation  du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.
**/

#include "qSelectCouleur.h"

#include <QPixmap>
#include <QSpacerItem>
#include <QMouseEvent>

#ifdef QSC_CAPTURE_PIXEL_ECRAN
#include <QScreen>
#include <QWindow>
#include <QGuiApplication>
#endif // QSC_CAPTURE_PIXEL_ECRAN

#ifdef DEBUG
#include <QDebug>
#endif // DEBUG

#define TAILLE_MATRICE_PIXEL 256
#define TAILLE_RECT_SELECTION_Y 32
#define TAILLE_BORDURE 3
#define TAILLE_SEPARATION_RECTS 3

#define TAILLE_IMAGE_X ( TAILLE_MATRICE_PIXEL+TAILLE_BORDURE+TAILLE_BORDURE )
#define TAILLE_IMAGE_Y ( TAILLE_MATRICE_PIXEL+TAILLE_RECT_SELECTION_Y+TAILLE_BORDURE+TAILLE_BORDURE )

#define DIVIS_MINIATURE 2

#define TAILLE_MATRICE_PIXEL_MINIATURE ( TAILLE_MATRICE_PIXEL / DIVIS_MINIATURE )
#define TAILLE_RECT_SELECTION_Y_MINIATURE ( TAILLE_RECT_SELECTION_Y / DIVIS_MINIATURE )

#define TAILLE_IMAGE_X_MINIATURE ( TAILLE_MATRICE_PIXEL_MINIATURE+TAILLE_BORDURE+TAILLE_BORDURE )
#define TAILLE_IMAGE_Y_MINIATURE ( TAILLE_MATRICE_PIXEL_MINIATURE+TAILLE_RECT_SELECTION_Y_MINIATURE+TAILLE_BORDURE+TAILLE_BORDURE )

using namespace QSelectCouleur_priv;

const int DiffContraste = 3;

inline int calcPositionPixelImgRGB(const QImage &img, unsigned int x, unsigned int y)
{
    return ((y*img.bytesPerLine()) + (x*(img.depth()/8)));
}

inline void AssignePixelImgRBG32(uchar *bits, unsigned int pos, QRgb c)
{
    bits[pos] = qRed(c);
    bits[pos+1] = qGreen(c);
    bits[pos+2] = qBlue(c);
}

ConteneurImage256::ConteneurImage256(bool mode_minature, QWidget *parent) : QLabel(parent),
    mode_image_miniature(mode_minature), etat(true), etat_lecture_seule(false), pix_monochrome(false),
    image_couleurs((mode_minature ? TAILLE_IMAGE_X_MINIATURE : TAILLE_IMAGE_X),
                   (mode_minature ? TAILLE_IMAGE_Y_MINIATURE : TAILLE_IMAGE_Y), QImage::Format_RGB32),
    position_courante((mode_minature ? (TAILLE_IMAGE_X_MINIATURE/2) : (TAILLE_IMAGE_X/2))+TAILLE_BORDURE,
                      (mode_minature ? (TAILLE_IMAGE_Y_MINIATURE/2) : (TAILLE_IMAGE_Y/2))+TAILLE_BORDURE),
    intensite_bleu(TAILLE_IMAGE_X/2)
//    intensite_bleu((mode_minature ? (TAILLE_IMAGE_X_MINIATURE/2) : (TAILLE_IMAGE_X/2)))
{
    if (mode_minature)
    {
        setMaximumSize(QSize(TAILLE_IMAGE_X_MINIATURE, TAILLE_IMAGE_Y_MINIATURE));
    }
    else
    {
        setMaximumSize(QSize(TAILLE_IMAGE_X, TAILLE_IMAGE_Y));
    }

    image_couleurs.fill(Qt::black);
    Desactive();
}

void ConteneurImage256::AssigneIntensite(unsigned char val_bleu, bool monochrome, bool force_mise_a_jour)
{
    pix_monochrome = monochrome;
    if (monochrome)
    {
        if (mode_image_miniature)
        {
            position_courante.setX((val_bleu/DIVIS_MINIATURE)+TAILLE_BORDURE);
            position_courante.setY((val_bleu/DIVIS_MINIATURE)+TAILLE_BORDURE);
    //        intensite_bleu = val_bleu / DIVIS_MINIATURE;
        }
        else
        {
//            position_courante.setX(convCoord(val_bleu)+TAILLE_BORDURE+TAILLE_BORDURE);
//            position_courante.setY(convCoord(val_bleu)+TAILLE_BORDURE+TAILLE_BORDURE);
//            intensite_bleu = val_bleu;
            position_courante.setX((val_bleu)+TAILLE_BORDURE);
            position_courante.setY((val_bleu)+TAILLE_BORDURE);
        }
    }

    if (intensite_bleu != val_bleu || force_mise_a_jour)
    {
        intensite_bleu = val_bleu;
        Active();
    }
}

void ConteneurImage256::defCouleur(QRgb couleur, bool monochrome, bool force_mise_a_jour)
{
    pix_monochrome = monochrome;

    bool mise_a_jour_affichage = false;

    const int r = monochrome ? (mode_image_miniature ? ((qBlue(couleur)/DIVIS_MINIATURE)+TAILLE_BORDURE) :
                                                       qBlue(couleur)) :
                               (mode_image_miniature ? ((qRed(couleur)/DIVIS_MINIATURE)+TAILLE_BORDURE) :
                                                       (convCoord(qRed(couleur), true)+TAILLE_BORDURE+TAILLE_BORDURE));

    const int g = monochrome ? (mode_image_miniature ? ((qBlue(couleur)/DIVIS_MINIATURE)+TAILLE_BORDURE) :
                                                       qBlue(couleur)) :
                               (mode_image_miniature ? ((qGreen(couleur)/DIVIS_MINIATURE)+TAILLE_BORDURE) :
                                                       (convCoord(qGreen(couleur), true)+TAILLE_BORDURE+TAILLE_BORDURE));
    const int b = qBlue(couleur);

    if (r != position_courante.x())
    {
        position_courante.setX(r);
        mise_a_jour_affichage = true;
    }

    if (g != position_courante.y())
    {
        position_courante.setY(g);
        mise_a_jour_affichage = true;
    }

    if (b != intensite_bleu)
    {
        intensite_bleu = b;
        mise_a_jour_affichage = true;
    }

    if (mise_a_jour_affichage || force_mise_a_jour)
    {
        Active();
    }
}

void ConteneurImage256::defLectureSeule(bool lecture_seule)
{
    etat_lecture_seule = lecture_seule;
}

int ConteneurImage256::TailleMatrice() const
{
    if (mode_image_miniature)
        return TAILLE_MATRICE_PIXEL_MINIATURE;
    return TAILLE_MATRICE_PIXEL;
}

int ConteneurImage256::TailleRectSelection_Y() const
{
    if (mode_image_miniature)
        return TAILLE_RECT_SELECTION_Y_MINIATURE;
    return TAILLE_RECT_SELECTION_Y;
}

int ConteneurImage256::convCoord(int i, bool ignore_image_miniature) const
/* Convertit une coordonnée sur la matrice en son équivalent couleur. */
{
    const bool inverse = false;
    if (mode_image_miniature && !ignore_image_miniature)
    {
        return inverse ? TAILLE_MATRICE_PIXEL_MINIATURE-((i-TAILLE_BORDURE)*DIVIS_MINIATURE) : ((i-TAILLE_BORDURE)*DIVIS_MINIATURE);
    }
    return inverse ? TAILLE_MATRICE_PIXEL-(i-TAILLE_BORDURE) : (i-TAILLE_BORDURE);
}

void ConteneurImage256::GenImage(bool mise_a_jour_couleurs)
{
    const unsigned char val_bleu = intensite_bleu;
    const QPoint &position_repere = position_courante;
    const bool monochrome = pix_monochrome;

    const QRgb couleur_repere_blanc = qRgb(255,255,255);
    const QRgb couleur_repere_noir = qRgb(0,0,0);

    const QRgb couleur_neutre = qRgb(49,54,59);

    const unsigned int taille_matrice = TailleMatrice();
    const unsigned int taille_rect_selection_y = TailleRectSelection_Y();

    /* Génération de la palette: */
    if (mise_a_jour_couleurs)
    {
        if (Actif())
        {
            if (monochrome)
            {
                for(unsigned int iy=TAILLE_BORDURE; iy<taille_matrice+TAILLE_BORDURE; ++iy)
                {
                    QRgb *d = (QRgb *) image_couleurs.scanLine(iy);
                    for(unsigned int ix=TAILLE_BORDURE; ix<taille_matrice+TAILLE_BORDURE; ++ix)
                    {
//                        d[ix] = qRgb(val_bleu, val_bleu, val_bleu);
                        d[ix] = qRgb(convCoord(iy), convCoord(iy), convCoord(iy));
                    }
                }
            }
            else
            {
                for(unsigned int iy=TAILLE_BORDURE; iy<taille_matrice+TAILLE_BORDURE; ++iy)
                {
                    QRgb *d = (QRgb *) image_couleurs.scanLine(iy);
                    for(unsigned int ix=TAILLE_BORDURE; ix<taille_matrice+TAILLE_BORDURE; ++ix)
                    {
                        d[ix] = qRgb(convCoord(ix), convCoord(iy), val_bleu);
                    }
                }
            }
        }
        else
        {
            for(unsigned int iy=TAILLE_BORDURE; iy<taille_matrice+TAILLE_BORDURE; ++iy)
            {
                QRgb *d = (QRgb *) image_couleurs.scanLine(iy);
                for(unsigned int ix=TAILLE_BORDURE; ix<taille_matrice+TAILLE_BORDURE; ++ix)
                {
                    d[ix] = couleur_neutre;
                }
            }
        }
    }

//    qDebug() << "Position courante : " << position_repere << " : " << position_courante;

    if (Actif())
    {
        /* Remplissage du rectangle de sélection */
        if ((taille_rect_selection_y > TAILLE_SEPARATION_RECTS))
        {
//            QRgb couleur_courante = monochrome ? (qRgb(val_bleu, val_bleu, val_bleu)) : (qRgb(convCoord(position_repere.x()), convCoord(position_repere.y()), val_bleu));
            QRgb couleur_courante = monochrome ? (qRgb(convCoord(position_repere.y()), convCoord(position_repere.y()), convCoord(position_repere.y()))) :
                                                 (qRgb(convCoord(position_repere.x()), convCoord(position_repere.y()), val_bleu));

//            qDebug() << "Couleur courante : rgb(" << qRed(couleur_courante) << qGreen(couleur_courante) << qBlue(couleur_courante) << ")";

            const unsigned int fin_bordure = image_couleurs.height()-TAILLE_BORDURE;

            for(unsigned int iy=taille_matrice+TAILLE_BORDURE; iy<fin_bordure; ++iy)
            {
                QRgb *d = (QRgb *) image_couleurs.scanLine(iy);
                for(unsigned int ix=TAILLE_BORDURE; ix<taille_matrice+TAILLE_BORDURE; ++ix)
                {
                    d[ix] = ((iy < taille_matrice+TAILLE_BORDURE+TAILLE_SEPARATION_RECTS) ? (couleur_repere_noir) : (couleur_courante));
                }
            }
        }

        /* Ajout du repère. */
        if (!etat_lecture_seule)
        {
            image_rendu = image_couleurs;

//            QPoint position_repere_corrige = monochrome ? (QPoint(val_bleu, val_bleu)) : position_repere;
            QPoint position_repere_corrige = position_repere;

            if (position_repere_corrige.x() < TAILLE_BORDURE+2)
            {
                position_repere_corrige.setX(TAILLE_BORDURE+1);
            }
            else if (position_repere_corrige.x() >= int(taille_matrice+TAILLE_BORDURE-2))
            {
                position_repere_corrige.setX(taille_matrice+TAILLE_BORDURE-2);
            }

            if (position_repere_corrige.y() < TAILLE_BORDURE+2)
            {
                position_repere_corrige.setY(TAILLE_BORDURE+1);
            }
            else if (position_repere_corrige.y() >= int(taille_matrice+TAILLE_BORDURE-2))
            {
                position_repere_corrige.setY(taille_matrice+TAILLE_BORDURE-2);
            }

//            qDebug() << "position_repere_corrige : " << position_repere_corrige;

//            qDebug() << "Après correction : " << position_repere_corrige << " : " << position_repere;

            uchar *pixels_rendu = image_rendu.bits();

            for(unsigned int i=TAILLE_BORDURE; i<taille_matrice+TAILLE_BORDURE; ++i)
            {
                const unsigned int pos_lg = i;
                const unsigned int pos_x = position_repere_corrige.x();
                const unsigned int pos_y = position_repere_corrige.y();

                // Horizontal
                AssignePixelImgRBG32(pixels_rendu, calcPositionPixelImgRGB(image_rendu, pos_lg, pos_y),   couleur_repere_blanc);
                AssignePixelImgRBG32(pixels_rendu, calcPositionPixelImgRGB(image_rendu, pos_lg, pos_y-1), couleur_repere_noir);
                AssignePixelImgRBG32(pixels_rendu, calcPositionPixelImgRGB(image_rendu, pos_lg, pos_y+1), couleur_repere_noir);

                // Vertical
                AssignePixelImgRBG32(pixels_rendu, calcPositionPixelImgRGB(image_rendu, pos_x,  pos_lg),  couleur_repere_blanc);
                AssignePixelImgRBG32(pixels_rendu, calcPositionPixelImgRGB(image_rendu, pos_x-1, pos_lg), couleur_repere_noir);
                AssignePixelImgRBG32(pixels_rendu, calcPositionPixelImgRGB(image_rendu, pos_x+1, pos_lg), couleur_repere_noir);
            }

            AssignePixelImgRBG32(pixels_rendu, calcPositionPixelImgRGB(image_rendu, position_repere_corrige.x()-1, position_repere_corrige.y()),   couleur_repere_blanc);
            AssignePixelImgRBG32(pixels_rendu, calcPositionPixelImgRGB(image_rendu, position_repere_corrige.x()+1, position_repere_corrige.y()),   couleur_repere_blanc);
            AssignePixelImgRBG32(pixels_rendu, calcPositionPixelImgRGB(image_rendu, position_repere_corrige.x(),   position_repere_corrige.y()-1), couleur_repere_blanc);
            AssignePixelImgRBG32(pixels_rendu, calcPositionPixelImgRGB(image_rendu, position_repere_corrige.x(),   position_repere_corrige.y()+1), couleur_repere_blanc);
            setPixmap(QPixmap::fromImage(image_rendu));
        }
        else
        {
            setPixmap(QPixmap::fromImage(image_couleurs));
        }
    }
    else
    {
        setPixmap(QPixmap::fromImage(image_couleurs));
    }
}

QRgb ConteneurImage256::PixelCourant() const
{
    if (Actif())
    {
//        qDebug() << "  PixelCourant : " << position_courante;
        const QRgb couleur = (image_couleurs.pixel(position_courante.x(), position_courante.y()));
//        qDebug() << "PixelCourant() : " << "rgb(" << qRed(couleur) << ", " << qGreen(couleur) << ", " << qBlue(couleur) << ")";
        return couleur;
    }
    return qRgb(0,0,0);
}

void ConteneurImage256::defToolTip(const QString &s)
{
    setToolTip(s);
}

void ConteneurImage256::Active()
{
    etat = true;
    GenImage(true);
}

void ConteneurImage256::Desactive()
{
    etat = false;
    GenImage(true);
}

int ConteneurImage256::LargeurConteneur() const
{
    if (mode_image_miniature)
        return TAILLE_IMAGE_X_MINIATURE;
    return TAILLE_IMAGE_X;
}

int ConteneurImage256::HauteurConteneur() const
{
    if (mode_image_miniature)
        return TAILLE_IMAGE_Y_MINIATURE;
    return TAILLE_IMAGE_Y;
}

void ConteneurImage256::mousePressEvent(QMouseEvent *ev)
{
    if (etat_lecture_seule)
    {
        ev->accept();
        return;
    }

    QPoint pos = ev->pos();

    if (mode_image_miniature)
    {
        if (pos.x() < TAILLE_BORDURE)
            pos.setX(TAILLE_BORDURE);
        else if (pos.x() > (TAILLE_MATRICE_PIXEL_MINIATURE+TAILLE_BORDURE-1))
            pos.setX(TAILLE_MATRICE_PIXEL_MINIATURE+TAILLE_BORDURE-1);

        if (pos.y() < TAILLE_BORDURE)
            pos.setY(TAILLE_BORDURE);
        else if (pos.y() > (TAILLE_MATRICE_PIXEL_MINIATURE+TAILLE_BORDURE-1))
            pos.setY(TAILLE_MATRICE_PIXEL_MINIATURE+TAILLE_BORDURE-1);

//        pos.setX(pos.x()/DIVIS_MINIATURE);
//        pos.setY(pos.y()/DIVIS_MINIATURE);
    }
    else
    {
        if (pos.x() < TAILLE_BORDURE)
            pos.setX(TAILLE_BORDURE);
        else if (pos.x() > (TAILLE_MATRICE_PIXEL+TAILLE_BORDURE-1))
            pos.setX(TAILLE_MATRICE_PIXEL+TAILLE_BORDURE-1);

        if (pos.y() < TAILLE_BORDURE)
            pos.setY(TAILLE_BORDURE);
        else if (pos.y() > (TAILLE_MATRICE_PIXEL+TAILLE_BORDURE-1))
            pos.setY(TAILLE_MATRICE_PIXEL+TAILLE_BORDURE-1);
    }

//    qDebug() << "Sélection pixel : " << pos << " : " << ev->pos();

    if (!pix_monochrome)
    {
        position_courante.setX(pos.x());
    }
    else
    {
        position_courante.setX(pos.y());
    }
    position_courante.setY(pos.y());

    GenImage(false);
    emit SelectionPixel(PixelCourant());
    ev->accept();
}

void ConteneurImage256::mouseMoveEvent(QMouseEvent *ev)
{
    if (etat_lecture_seule)
    {
        ev->accept();
        return;
    }

    if (ev->buttons() & Qt::LeftButton)
    {
        mousePressEvent(ev);
    }
    ev->accept();
}

SelectCouleurMono::SelectCouleurMono(bool mode_miniature, QRgb type, QWidget *parent) : QWidget(parent), mode_image_miniature(mode_miniature)
{
    type_couleur = type;

    setLayout(&layout_select);
    layout_select.setSpacing(6);
    spb.setAlignment(Qt::AlignHCenter);

    const int n_min = 80; /* Niveau de couleur minimum sur la feuille de style */
    const int mult_c = 250-n_min;
    const QString couleur_str(QString::number((qRed(type_couleur)*mult_c)+n_min) + "," +
                              QString::number((qGreen(type_couleur)*mult_c)+n_min) + "," +
                              QString::number((qBlue(type_couleur)*mult_c)+n_min));
    spb.setStyleSheet(QString("color: rgb(") + couleur_str + ");");

    lbl.setMinimumSize(QSize(26, 16));
    lbl.setMaximumSize(QSize(26, 16));

    spb.setRange(0, 255);
    spb.setMinimumWidth(95);
    layout_select.addWidget(&spb);
    layout_select.addWidget(&lbl);

    connect(&spb, SIGNAL(valueChanged(int)), this, SLOT(ModifCouleur(int)));
}

void SelectCouleurMono::defValCouleur(int val, bool desactive_mise_a_jour)
{
    if (val != spb.value())
    {
        desactive_mise_a_jour_ = desactive_mise_a_jour;
//        val = (val < 0) ? 0 : ((val > 255) ? 255 : val);
        spb.setValue(val);
    }
}

int SelectCouleurMono::valCouleur() const
{
    return spb.value();
}

void SelectCouleurMono::defToolTip(const QString &s)
{
    spb.setToolTip(s);
}

void SelectCouleurMono::ModifCouleur(int val)
{
    QPixmap img(32, 16);
    const QColor couleur(qRed(type_couleur)*val, qGreen(type_couleur)*val, qBlue(type_couleur)*val);
    img.fill(couleur);
    lbl.setPixmap(img);

//    if (desactive_mise_a_jour_)
//    {
//        desactive_mise_a_jour_ = false;
//        return;
//    }

//    qDebug() << "defValCouleur :: " << couleur << " (" << val << ")";

    emit SelectionValeur(val);
}

QSelectCouleur::QSelectCouleur(bool mode_miniature, QWidget *parent) : QWidget(parent), image(mode_miniature), couleur_sav(0),
    active_selecteurs(mode_miniature ? false : false /* A débugger?! */), layout_v(this),
    select_rouge(mode_miniature, qRgb(1,0,0)), select_vert(mode_miniature, qRgb(0,1,0)), select_bleu(mode_miniature, qRgb(0,0,1)),
    slider_v(Qt::Vertical)
{
    /* Initialisation de l'image: */
//    GenImage(0);

    layout_v.setSpacing(0);
    layout_v.setSizeConstraint(QLayout::SetMinimumSize);
    layout_v.setContentsMargins(0,0,0,0);

    layout_h.setSizeConstraint(QLayout::SetMinimumSize);
    layout_h.setSpacing(24);
    layout_h.setContentsMargins(0,0,0,0);

//    this->setLayout(&layout_v);

    layout_v.addStretch(0);
    layout_v.addLayout(&layout_h);
    layout_v.addStretch(0);

    layout_h.addStretch(0);

    layout_image.addWidget(&image);

    layout_image.addLayout(&layout_intensite);
    layout_intensite.setContentsMargins(TAILLE_BORDURE, 0, TAILLE_BORDURE, 0);
    slider_v.setMinimum(0);
    slider_v.setInvertedControls(true);
    slider_v.setInvertedAppearance(true);
    slider_v.setTracking(true);
    slider_v.setEnabled(false);
//    slider_v.setMaximumHeight(image.HauteurConteneur());

    layout_intensite.addWidget(&slider_v);

    if (mode_miniature)
    {
        slider_v.setMaximumHeight(TAILLE_MATRICE_PIXEL_MINIATURE);
//        slider_v.setMaximum(TAILLE_MATRICE_PIXEL-1);
        layout_intensite.addSpacing(TAILLE_RECT_SELECTION_Y_MINIATURE-TAILLE_BORDURE);
    }
    else
    {
        slider_v.setMaximumHeight(TAILLE_MATRICE_PIXEL);
//        slider_v.setMaximum(TAILLE_MATRICE_PIXEL-1);
        layout_intensite.addSpacing(TAILLE_RECT_SELECTION_Y-TAILLE_BORDURE);
    }

    slider_v.setMaximum(TAILLE_MATRICE_PIXEL-1);

    slider_v.setValue(slider_v.maximum()/2);

    layout_palette.addLayout(&layout_image);
    layout_h.addLayout(&layout_palette);

    slider_v.setToolTip(tr("Sélection de la couleur (niveau de bleu)"));
    connect(&slider_v, SIGNAL(valueChanged(int)), this, SLOT(changeCouleur(int)));

    image.defToolTip(tr("Sélection de la couleur (niveaux de rouge et de vert)"));
    connect(&image, SIGNAL(SelectionPixel(QRgb)), this, SLOT(SelectionPixel(QRgb)));

    layout_select_couleurs.setSpacing(6);

    bouton_restaurer_couleur.setText(tr("Restaurer"));
    bouton_restaurer_couleur.setToolTip(tr("Restaure la couleur d'origine"));
    bouton_restaurer_couleur.setIcon(QIcon(":/icones/annuler.png"));
    bouton_restaurer_couleur.setEnabled(false);
    layout_select_couleurs.addWidget(&bouton_restaurer_couleur);
    connect(&bouton_restaurer_couleur, SIGNAL(pressed()), this, SLOT(Clic_restaure_couleur()));

    if (!mode_miniature)
    {
#ifdef QSC_CAPTURE_PIXEL_ECRAN
        bouton_capture_pixel_ecran.setText(tr("Capturer..."));
        bouton_capture_pixel_ecran.setToolTip(tr("Capturer une couleur depuis un pixel affiché sur l'écran"));
        bouton_capture_pixel_ecran.setIcon(QIcon(":/icones/viseur.png"));
        bouton_capture_pixel_ecran.setEnabled(false);
        layout_select_couleurs.addWidget(&bouton_capture_pixel_ecran);
        connect(&bouton_capture_pixel_ecran, SIGNAL(pressed()), this, SLOT(Clic_capture_pixel_ecran()));
#endif // QSC_CAPTURE_PIXEL_ECRAN
    }

    if (!mode_miniature)
    {
        layout_select_couleurs.addSpacing(32);
    }

    chk_monochrome.setText(tr("Monochrome"));
    chk_monochrome.setToolTip(tr("Mode monochrome"));
    chk_monochrome.setChecked(false);
    chk_monochrome.setEnabled(false);
    chk_monochrome.setTristate(false);
    layout_select_couleurs.addWidget(&chk_monochrome);
    connect(&chk_monochrome, SIGNAL(toggled(bool)), this, SLOT(changeEtatMonochrome(bool)));

    if (active_selecteurs)
    {
        layout_select_couleurs.addSpacing(12);
    }
    else
    {
        layout_select_couleurs.addStretch(0);
    }

    if (active_selecteurs)
    {
        bouton_contraste_moins.setObjectName("mini");
        bouton_contraste_moins.setIcon(QIcon(":/icones/b_moins_c"));
        bouton_contraste_moins.setText("");
        bouton_contraste_moins.setToolTip(tr("Réduire le contraste"));
        bouton_contraste_moins.setAutoRepeat(true);
        bouton_contraste_moins.setAutoRepeatDelay(500);

        bouton_contraste_plus.setObjectName("mini");
        bouton_contraste_plus.setIcon(QIcon(":/icones/b_plus_c"));
        bouton_contraste_plus.setText("");
        bouton_contraste_plus.setToolTip(tr("Monter le contraste"));
        bouton_contraste_plus.setAutoRepeat(true);
        bouton_contraste_plus.setAutoRepeatDelay(500);
        layout_boutons_contraste.addSpacing(28);

        layout_boutons_contraste.addWidget(&bouton_contraste_moins);
        layout_boutons_contraste.addWidget(&bouton_contraste_plus);
        layout_boutons_contraste.addStretch();
        connect(&bouton_contraste_moins, SIGNAL(pressed()), this, SLOT(BaisseContraste()));
        connect(&bouton_contraste_plus, SIGNAL(pressed()), this, SLOT(MonteContraste()));
        layout_select_couleurs.addLayout(&layout_boutons_contraste);

        select_rouge.defToolTip(tr("Niveau de rouge"));
        select_vert.defToolTip(tr("Niveau de vert"));
        select_bleu.defToolTip(tr("Niveau de bleu"));

        select_rouge.setEnabled(false);
        select_vert.setEnabled(false);
        select_bleu.setEnabled(false);

        layout_select_couleurs.addWidget(&select_rouge);
        layout_select_couleurs.addWidget(&select_vert);
        layout_select_couleurs.addWidget(&select_bleu);
        layout_select_couleurs.addStretch(6);
        connect(&select_rouge, SIGNAL(SelectionValeur(int)), this, SLOT(SelectionRouge(int)));
        connect(&select_vert, SIGNAL(SelectionValeur(int)), this, SLOT(SelectionVert(int)));
        connect(&select_bleu, SIGNAL(SelectionValeur(int)), this, SLOT(SelectionBleu(int)));
    }

    layout_h.addLayout(&layout_select_couleurs);

    layout_h.addStretch(0);
}

void QSelectCouleur::changeEtatMonochrome(bool etat)
{
    if (active_selecteurs)
    {
        select_rouge.setEnabled(!etat);
        select_vert.setEnabled(!etat);
        select_bleu.setEnabled(!etat);
    }
    image.AssigneIntensite((unsigned char) slider_v.value(), etat, true);
//    qDebug() << "changeEtatMonochrome :: " << etat << " : " << QColor(image.PixelCourant());
//    SelectionPixel(image.PixelCourant());
}

void QSelectCouleur::changeCouleur(int val)
{
    image.AssigneIntensite((unsigned char) val, chk_monochrome.isChecked());
    SelectionPixel(image.PixelCourant());
}

void QSelectCouleur::defCouleur(QRgb couleur)
{
    couleur_sav = couleur;
    bouton_restaurer_couleur.setEnabled(true);

#ifdef QSC_CAPTURE_PIXEL_ECRAN
    bouton_capture_pixel_ecran.setEnabled(true);
#endif // QSC_CAPTURE_PIXEL_ECRAN

    defCouleurInt(couleur, false, true);
}


void QSelectCouleur::defCouleurInt(QRgb couleur, bool ignore_mono_chrome, bool force_mise_a_jour)
{
    chk_monochrome.setEnabled(true);
    slider_v.setEnabled(true);

    if ((qRed(couleur) == select_rouge.valCouleur()) && (qGreen(couleur) == select_vert.valCouleur()) && (qBlue(couleur) == select_bleu.valCouleur()))
    {
        return;
    }

//    qDebug() << "   defCouleurInt : " << qRed(couleur) << ":" << qGreen(couleur) << ":" << qBlue(couleur) << " :: " << select_rouge.valCouleur() << ":" << select_vert.valCouleur() << ":" << select_bleu.valCouleur();

    if (!ignore_mono_chrome && (qRed(couleur) == qGreen(couleur)) && (qRed(couleur) == qBlue(couleur)) )
    {
        chk_monochrome.setChecked(true);

        if (force_mise_a_jour)
        {
            image.defCouleur(couleur, true, force_mise_a_jour);
        }

        if (active_selecteurs)
        {
            select_rouge.setEnabled(false);
            select_vert.setEnabled(false);
            select_bleu.setEnabled(false);
        }

        slider_v.setValue(qGreen(couleur)); // Equivalent coordonnée Y
//        qDebug() << "slider_v.value() :: " << slider_v.value();
    }
    else
    {
        chk_monochrome.setChecked(false);

        if (force_mise_a_jour)
        {
            image.defCouleur(couleur, false, force_mise_a_jour);
        }

        if (active_selecteurs)
        {
            select_rouge.setEnabled(true);
            select_vert.setEnabled(true);
            select_bleu.setEnabled(true);
            select_rouge.defValCouleur(qRed(couleur), true);
            select_vert.defValCouleur(qGreen(couleur), true);
            select_bleu.defValCouleur(qBlue(couleur), false);
        }
        slider_v.setValue(qBlue(couleur));
//        qDebug() << "slider_v.value() :: " << slider_v.value();
    }

//    slider_v.setValue(qBlue(couleur));
}


bool QSelectCouleur::RestaureCouleur()
{
    if (couleur_sav != static_cast<QRgb>(0))
    {
        defCouleur(couleur_sav);
        return true;
    }
    return false;
}

QRgb QSelectCouleur::Couleur() const
{
    return image.PixelCourant();
}

int QSelectCouleur::HauteurWidget() const
{
    return image.HauteurConteneur();
}

void QSelectCouleur::SelectionPixel(QRgb couleur)
{
//    qDebug() << "Sélection : " << couleur << " : rgb(" << qRed(couleur) << qGreen(couleur) << qBlue(couleur) << ")";

    if (slider_v.value() != qBlue(couleur))
    {
        slider_v.setValue(qBlue(couleur));
//        qDebug() << "slider_v.value() :: " << slider_v.value();
        return; /* Un nouveau signal sera émis */
    }

    if (active_selecteurs)
    {
        select_rouge.defValCouleur(qRed(couleur), true);
        select_vert.defValCouleur(qGreen(couleur), true);
        select_bleu.defValCouleur(qBlue(couleur), false);
    }

    emit NouvelleCouleur(couleur);
}

void QSelectCouleur::SelectionRouge(int val)
{
    if (!active_selecteurs)
    {
        return;
    }
    if (chk_monochrome.isChecked())
    {
        return;
    }

//    if (val != select_rouge.valCouleur())
    {
        const QRgb couleur = qRgb(val, select_vert.valCouleur(), select_bleu.valCouleur());
        defCouleurInt(couleur, true, false);
        SelectionPixel(couleur);
    }
}

void QSelectCouleur::SelectionVert(int val)
{
    if (!active_selecteurs)
    {
        return;
    }
    if (chk_monochrome.isChecked())
    {
        return;
    }

//    if (val != select_vert.valCouleur())
    {
        const QRgb couleur = qRgb(select_rouge.valCouleur(), val, select_bleu.valCouleur());
        defCouleurInt(couleur, true, false);
        SelectionPixel(couleur);
    }
}

void QSelectCouleur::SelectionBleu(int val)
{
    if (!active_selecteurs)
    {
        return;
    }
//    if (chk_monochrome.isChecked())
//    {
////        slider_v.setValue(val);
//        return;
//    }
//    changeCouleur(val);

//    if (val != select_bleu.valCouleur())
    {
        const QRgb couleur = qRgb(select_rouge.valCouleur(), select_vert.valCouleur(), val);
        defCouleurInt(couleur, !chk_monochrome.isChecked(), false);
        changeCouleur(val);
    }
}

void QSelectCouleur::BaisseContraste()
{
    if (active_selecteurs)
    {
        select_rouge.defValCouleur(select_rouge.valCouleur()-DiffContraste, true);
        select_vert.defValCouleur(select_vert.valCouleur()-DiffContraste, true);
        select_bleu.defValCouleur(select_bleu.valCouleur()-DiffContraste, false);
    }
}

void QSelectCouleur::MonteContraste()
{
    if (active_selecteurs)
    {
        select_rouge.defValCouleur(select_rouge.valCouleur()+DiffContraste, true);
        select_vert.defValCouleur(select_vert.valCouleur()+DiffContraste, true);
        select_bleu.defValCouleur(select_bleu.valCouleur()+DiffContraste, false);
    }
}

void QSelectCouleur::Clic_restaure_couleur()
{
    RestaureCouleur();
}

#ifdef QSC_CAPTURE_PIXEL_ECRAN
QRgb QSelectCouleur::CapturePixelEcran(const QPoint &pos) const
/* Capture et renvoi la couleur d'un pixel à l'écran. */
{
    QScreen *ecran = Q_NULLPTR;

//#if QT_VERSION >= QT_VERSION_CHECK(5, 8, 0)
    QWidget *parent_w = this->window();
    while (true)
    {
        QWidget *parent_w_tmp = parent_w->parentWidget();
        if (parent_w_tmp == Q_NULLPTR)
        {
            break;
        }
        parent_w = parent_w_tmp;
    }
//#endif // QT_VERSION

    if (parent_w != Q_NULLPTR)
    {
        const QWindow *win = parent_w->windowHandle();
        ecran = win->screen();
    }

    if (ecran == Q_NULLPTR)
    {
        ecran = QGuiApplication::primaryScreen();
        if (const QWindow *win = windowHandle())
        {
            ecran = win->screen();
        }
    }

    if (ecran == Q_NULLPTR)
    {
#ifdef DEBUG
//        qDebug() << "QSelectCouleur::CapturePixelEcran(" << pos << ") :: Attention, impossible de déterminer l'écran principal !";
#endif // DEBUG
        return qRgb(0,0,0);
    }
    const QPixmap pixmap = ecran->grabWindow(0, pos.x(), pos.y(), 1, 1);
    QImage i = pixmap.toImage();
    const QRgb rgb = i.pixel(0, 0);

#ifdef DEBUG
//    qDebug() << "QSelectCouleur::CapturePixelEcran(" << pos << "/" << ecran->availableSize() << ") :: [" << i.width() << ":" << i.height() << "] :: " << QColor(rgb);
#endif // DEBUG

    return rgb;
}

void QSelectCouleur::TermineCapturePixelEcran()
{
//    qDebug() << "Fin de la capture du pixel écran...";

    bouton_capture_pixel_ecran.setEnabled(true);
    this->releaseMouse();
    this->releaseKeyboard();
    setMouseTracking(false);
    QGuiApplication::setOverrideCursor(QCursor(Qt::ArrowCursor));
    image.defLectureSeule(false);
    etat_capture_pixel_ecran = false;
}

void QSelectCouleur::CommenceCapturePixelEcran()
{
//    qDebug() << "Début de la capture du pixel écran : " << etat_capture_pixel_ecran;

    etat_capture_pixel_ecran = true;
    this->grabMouse();
    this->grabKeyboard();
    QGuiApplication::setOverrideCursor(QCursor(Qt::CrossCursor));
    setMouseTracking(true);
    image.defLectureSeule(true);
    bouton_capture_pixel_ecran.setEnabled(false);
}

void QSelectCouleur::mouseMoveEvent(QMouseEvent *e)
{
    if (etat_capture_pixel_ecran)
    {
//        image.defCouleur(CapturePixelEcran(e->globalPos()), false);
        defCouleurInt(CapturePixelEcran(e->globalPos()), true, true);
        e->accept();
        return;
    }
    QWidget::mouseMoveEvent(e);
}

void QSelectCouleur::mousePressEvent(QMouseEvent *e)
{
    if (etat_capture_pixel_ecran)
    {
        TermineCapturePixelEcran();

        if (e->buttons() & Qt::LeftButton)
        {
            const QRgb couleur = CapturePixelEcran(e->globalPos());
//            qDebug() << "Capture : " << couleur;
            defCouleurInt(couleur, false, true);
//            emit NouvelleCouleur(couleur);
        }

        e->accept();
        return;
    }
    QWidget::mouseReleaseEvent(e);
}

void QSelectCouleur::keyPressEvent(QKeyEvent *e)
{
    if (etat_capture_pixel_ecran)
    {
//        if (e->key() == Qt::Key_Escape)
        {
            TermineCapturePixelEcran();
        }
        e->accept();
        return;
    }
    QWidget::keyPressEvent(e);
}

void QSelectCouleur::Clic_capture_pixel_ecran()
{
    CommenceCapturePixelEcran();
}
#endif // QSC_CAPTURE_PIXEL_ECRAN
