﻿/**
© Florian Joncour, 2013-2018 florian@zetta-sys.com

Ce logiciel est un programme informatique faisant interface à la bibliothèque
Perspective3D, un outil de modélisation 3D à partir de vues orthographiques 2D.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site "http://www.cecill.info".

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée.  Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme,  le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard  l'attention de l'utilisateur est attirée sur les risques
associés au chargement,  à l'utilisation,  à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant  des  connaissances  informatiques approfondies.  Les
utilisateurs sont donc invités à charger  et  tester  l'adéquation  du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.
**/

#include "qAudioWave.h"
#include "perspective_sys.h"

#include <QtGlobal>
#include <QIODevice>
#include <QAudioOutput>
#include <QAudioDeviceInfo>
// #include <QAudioFormat>
#include <QFile>

#include <QMutex>
#include <QThread>

#include <string.h>

#ifdef DEBUG
#include <QDebug>
#endif

union int16_bt
{
    qint16 i;
    char b[sizeof(qint16)];
};

union int32_bt
{
    qint32 i;
    char b[sizeof(qint32)];
};

inline qint16 ChargeBin16(char b0, char b1)
{
    int16_bt i;
    if (PerspectiveSys::PetitBoutisme())
    {
        i.b[0] = b0;
        i.b[1] = b1;
    }
    else
    {
        i.b[1] = b0;
        i.b[0] = b1;
    }
    return i.i;
}

inline qint32 ChargeBin32(char b0, char b1, char b2, char b3)
{
    int32_bt i;

    if (PerspectiveSys::PetitBoutisme())
    {
        i.b[0] = b0;
        i.b[1] = b1;
        i.b[2] = b2;
        i.b[3] = b3;
    }
    else
    {
        i.b[3] = b0;
        i.b[2] = b1;
        i.b[1] = b2;
        i.b[0] = b3;
    }
    return i.i;
}

namespace ExecAudio
{
    int ExecAudioPCM(const char *tampon_fichier, unsigned long debut_lecture_tampon, unsigned long fin_lecture_tampon, unsigned long taille_periode, QIODevice *fichier_audio, bool *arret)
    {
        if (!fichier_audio)
        {
            return ERR_IO;
        }

        int taux_erreurs = 0;
        const char *ptr_tampon = tampon_fichier + debut_lecture_tampon;
        const int taille_tampon = fin_lecture_tampon - debut_lecture_tampon;
        const int taille_ecriture = taille_periode; // 4096
        const int seuil_erreurs = 12; //taille_tampon / (taille_ecriture);

        int pos_ptr = 0;

        if (!taille_ecriture)
        {
            fichier_audio->close();
            return ERR_TAMPON_VIDE;
        }

//        qDebug() << "Seuil d'erreurs : " << seuil_erreurs;

        while (pos_ptr < taille_tampon)
        {
            const int taille_ec = (pos_ptr+taille_ecriture < taille_tampon) ? taille_ecriture : taille_tampon-pos_ptr;
            int etat_ecriture = fichier_audio->write(ptr_tampon+pos_ptr, taille_ec);

//            qDebug() << ":: " << pos_ptr << "/" << taille_tampon << "(" << etat_ecriture << "/" << taille_ec << ")";

            if (arret != nullptr)
            {
                if (*arret)
                {
                    return ERR_ARRET;
                }
            }

            if (etat_ecriture < 0)
            {
                fichier_audio->close();
                return ERR_IO;
            }
            else if (etat_ecriture == 0) /* Rien n'a été écrit. Périphérique occupé ? */
            {
#ifdef DEBUG
//                qDebug() << "ExecAudioPCM :: Rien à écrire ? " << taux_erreurs << "/" << seuil_erreurs;
//                qDebug() << "   " << fichier_audio->errorString();
#endif
                if (taux_erreurs > seuil_erreurs)
                {
#ifdef DEBUG
//                    qDebug() << "ExecAudioPCM :: Arrêt de la lecture à la suite d'erreurs " << taux_erreurs << "/" << seuil_erreurs;
#endif
                    fichier_audio->close();
                    return ERR_OCCUPE;
                }
                QThread::msleep(50);
                ++taux_erreurs;
            }
            else /* Tout est normal. */
            {
                if (etat_ecriture != taille_ec) /* Il manque des données ?! */
                {
#ifdef DEBUG
//                    qDebug() << "qAudioWave::Lecture :: " << etat_ecriture << "!=" << taille_ec;
#endif
                }
                pos_ptr += etat_ecriture;
                taux_erreurs = 0;

//                const unsigned int tempo = (debit_octs_secs) / (etat_ecriture*1000);
//                qDebug() << "Temporisation : " << tempo << "=" << debit_octs_secs << "/" << etat_ecriture;

//                QThread::usleep(tempo);
            }
        }

        const int taille_ecriture_fin = taille_tampon - pos_ptr;
        //     qDebug() << "qAudioW : Nombre d'informations restantes à écrire : " << taille_ecriture_fin << " (" << pos_ptr << "/" << taille_tampon << ")";

        if (taille_ecriture_fin > 0)
        {
#ifdef DEBUG
            qDebug() << "qAudioWave : Nombre d'informations restantes à écrire : " << taille_ecriture_fin << " (" << pos_ptr << "/" << taille_tampon << ")";
#endif
            int etat_ecriture = fichier_audio->write(ptr_tampon+pos_ptr, taille_ecriture_fin); /* Termine l'ecriture ... */
            P_UNUSED(etat_ecriture);
//            qDebug() << "Reste : " << taille_ecriture_fin-etat_ecriture;
        }
        else
        {
#ifdef DEBUG
//            qDebug() << "qAudioW : Tout a été envoyé : " << taille_tampon << ")";
#endif
        }

        fichier_audio->close();
        return ERR_VALIDE;
    }

    ExecAudio_th::ExecAudio_th(const char *tampon_fichier_, unsigned long debut_lecture_tampon_, unsigned long fin_lecture_tampon_, unsigned long taille_periode_, QIODevice *fichier_audio_) :
        tampon_fichier(tampon_fichier_), debut_lecture_tampon(debut_lecture_tampon_), fin_lecture_tampon(fin_lecture_tampon_), taille_periode(taille_periode_), fichier_audio(fichier_audio_),
        etat_lecture(-1), arret_lecture(false)
    {
        ;
    }

    bool ExecAudio_th::EnCours() const
    {
        return etat_lecture == -1;
    }

    void ExecAudio_th::run()
    {
        etat_lecture = -1;
        arret_lecture = false;
        int etat = ExecAudioPCM(tampon_fichier, debut_lecture_tampon, fin_lecture_tampon, taille_periode, fichier_audio, &arret_lecture);
        if (etat != ERR_ARRET)
        {
            emit FinLecture(etat, QAudio::NoError);
        }
        etat_lecture = etat;
    }

    void ExecAudio_th::Stop()
    {
        arret_lecture = true;
        QThread::msleep(55);
    }

    void ExecAudio_th::AttendFinLecture() const
    {
        while (EnCours())
        {
            QThread::msleep(15);
        }
    }
}

qAudioWave::qAudioWave() : valide_chargement(false), valide_lecture(false), audio(nullptr), fichier_audio(nullptr), th_lecture(nullptr), volume(0.75)
{
}

qAudioWave::qAudioWave(const QString chemin_fichier) : valide_chargement(false), valide_lecture(false), audio(nullptr), fichier_audio(nullptr), th_lecture(nullptr), volume(0.75)
{
    Charge(chemin_fichier);
}

qAudioWave::~qAudioWave()
{
}

bool qAudioWave::Charge(const QString &chemin)
/* Chargement d'un fichier WAVE. Renvoi true si le format est supporté et prêt à être lu. Sinon false. */
{
    valide_chargement = false;
    valide_lecture = false;
    audio = nullptr;
    debut_donnes = fin_donnes = debit_octs_secs = 0;
    mesg_erreur.clear();

    /* Lecture de l'entête: */
    QFile f(chemin);

    if (!f.exists())
    {
        AjoutErreur(tr("Impossible d'accéder au fichier") + " " + chemin);
        return false;
    }

    if (!f.open(QIODevice::ReadOnly))
    {
        AjoutErreur(tr("Impossible d'ouvrir le fichier") + " " + chemin);
        return false;
    }

    donnes = f.readAll();
    f.close();

    if (donnes.isEmpty())
    {
        AjoutErreur(tr("Fichier %1 vide").arg(chemin));
        return false;
    }

    const char *tampon = donnes.data();

    /* Chargement du format (44 premiers octets du fichier). */
    char RIFF[5];
    char WAVE[5];
    char FMT[5];
    char DATA[5];

    if (!(tampon[0] == 'R' && tampon[1] == 'I' && tampon[2] == 'F' && tampon[3] == 'F'))
    {
        strncpy(RIFF, tampon, 4);
        RIFF[4] = '\0';
        AjoutErreur(tr("Format non valide") + " (" + RIFF + "!=" + "RIFF)");
        return false;
    }

    qint32 taille_fichier = ChargeBin32(tampon[4], tampon[5], tampon[6], tampon[7]);
//     qDebug() << "Taille du fichier -8 : " << taille_fichier;

    if (taille_fichier+8 != donnes.size())
    {
        AjoutErreur(tr("Décalage données (taille du fichier)") + " (" + QString::number(taille_fichier+8) + " != " + QString::number(donnes.size()) + " )");
        return false;
    }

    if (!(tampon[8] == 'W' && tampon[9] == 'A' && tampon[10] == 'V' && tampon[11] == 'E'))
    {
        strncpy(WAVE, tampon+8, 4);
        WAVE[4] = '\0';
        AjoutErreur(tr("Format non valide") + " (" + WAVE + "!=" + "WAVE)");
        return false;
    }

    if (!(tampon[12] == 'f' && tampon[13] == 'm' && tampon[14] == 't' && tampon[15] == ' '))
    {
        strncpy(FMT, tampon+12, 4);
        FMT[4] = '\0';
        AjoutErreur(tr("Format non valide") + " (" + FMT + "!=" + "fmt )");
        return false;
    }

    qint32 taille_bloc = ChargeBin32(tampon[16], tampon[17], tampon[18], tampon[19]);

    if (taille_bloc != 16)
    {
        AjoutErreur(tr("Taille du bloc invalide") + " (" + QString::number(taille_bloc) + "!=16)");
        return false;
    }

//     qDebug() << "Taille du bloc : " << taille_bloc;

    qint16 format_audio = ChargeBin16(tampon[20], tampon[21]);

    if (format_audio != 1) /* On accepte uniquement le format PCM */
    {
        AjoutErreur(tr("Format invalide") + " (" + QString::number(format_audio) + "!=1 (PCM))");
        return false;
    }

    qint16 nombre_canaux = ChargeBin16(tampon[22], tampon[23]);

    if (nombre_canaux < 0 || nombre_canaux > 6)
    {
        AjoutErreur(tr("Nombre de canaux invalide") + " (" + QString::number(nombre_canaux) + ")");
        return false;
    }

//     qDebug() << "Nombre de canaux : " << nombre_canaux;

    qint32 frequence_echantillons = ChargeBin32(tampon[24], tampon[25], tampon[26], tampon[27]);

    if (frequence_echantillons != 11025 && frequence_echantillons != 22050 && frequence_echantillons != 44100 && frequence_echantillons != 48000 && frequence_echantillons != 96000)
    {
        AjoutErreur(tr("Fréquence d'échantillonage invalide") + " (" + QString::number(frequence_echantillons) + ")");
        return false;
    }
//     qDebug() << "Fréquence d'échantillonage : " << frequence_echantillons;

    qint32 octets_secondes = ChargeBin32(tampon[28], tampon[29], tampon[30], tampon[31]);
//     qint16 octets_blocs = ChargeBin16(tampon[32], tampon[33]);
    qint16 octets_echantillons = ChargeBin16(tampon[34], tampon[35]);

//     qDebug() << "Octets par secondes : " << octets_secondes;
//     qDebug() << "Octets par blocs : " << octets_blocs;
//     qDebug() << "Octets par échantillon : " << octets_echantillons;

    if (octets_echantillons != 8 && octets_echantillons != 16 && octets_echantillons != 24)
    {
        AjoutErreur(tr("Nombre d'octets par échantillon invalide") + " (" + QString::number(octets_echantillons) + ")");
        return false;
    }

    /* A partir de là, l'entête doit être valide. */
    format.setSampleRate(frequence_echantillons);
    format.setChannelCount(nombre_canaux);
    format.setSampleSize(octets_echantillons);
    format.setCodec("audio/pcm");
    format.setByteOrder(QAudioFormat::LittleEndian);

    if (octets_echantillons == 8)
    {
        format.setSampleType(QAudioFormat::UnSignedInt);
    }
    else
    {
        format.setSampleType(QAudioFormat::SignedInt);
    }

    if (!format.isValid())
    {
        AjoutErreur(tr("Format non valide"));
        return false;
    }

    if (!(tampon[36] == 'd' && tampon[37] == 'a' && tampon[38] == 't' && tampon[39] == 'a'))
    {
        strncpy(DATA, tampon+36, 4);
        DATA[4] = '\0';
        AjoutErreur(tr("Décalage dans l'entête") + " (" + DATA + "!=" + "data)");
        return false;
    }

    qint32 taille_donnes = ChargeBin32(tampon[40], tampon[41], tampon[42], tampon[43]);
//     int taille_extension = (donnes.size() - 44) - taille_donnes; /* Informations id3 (seront ignorées à la lecture. */
//     qDebug() << "Taille des données : " << taille_donnes << ", extension : " << taille_extension;

    debut_donnes = 44;
    fin_donnes = debut_donnes + (taille_donnes);
    debit_octs_secs = octets_secondes;

//     qDebug() << "Lecture PCM de " << debut_donnes << " à " << fin_donnes;

    valide_chargement = true;
    return valide_chargement;
}

bool qAudioWave::LectureEnCours() const
{
    if (th_lecture)
    {
        return th_lecture->EnCours();
    }
    return false;
}

bool qAudioWave::Lecture(const QAudioDeviceInfo *peripherique, qreal volume_r, bool bloquant)
{
    mesg_erreur.clear();

    if (!peripherique)
    {
        AjoutErreur(tr("Périphérique invalide."));
        return false;
    }

    if (peripherique->isNull())
    {
        AjoutErreur(tr("Périphérique invalide."));
        return false;
    }

    return ExecLecture(peripherique, (volume_r > 0) ? volume_r : volume, bloquant);
}

bool qAudioWave::ExecLecture(const QAudioDeviceInfo *peripherique, qreal volume_r, bool bloquant)
{
    valide_lecture = false;

    if (!peripherique->isFormatSupported(format))
    {
        AjoutErreur(tr("Format audio de sortie non supporté par le périphérique.\nLe périphérique est peut être occupé."));
        return false;
    }

    if (!ValideChargement())
    {
        AjoutErreur(tr("Rien à lire"));
        return false;
    }

    if (!Nettoyage())
    {
#ifdef DEBUG
        AjoutErreur(tr("Supperposition de lecture"));
        return false;
#else
        return true; /* On quitte dans ce cas là gentiement */
#endif
    }

    audio = new QAudioOutput(*peripherique, format);
    audio->setBufferSize(audio->periodSize());

    fichier_audio = audio->start();
    if (audio->state() != QAudio::ActiveState && audio->state() != QAudio::IdleState)
    {
#ifdef DEBUG
        qDebug() << "Erreur d'initialisation audio : " << audio->state();
#endif
        AjoutErreur(tr("Erreur I/O (initialisation). Impossible de démarrer le périphérique"));
        return false;
    }

    audio->setVolume(volume_r);

    if (!fichier_audio)
    {
        AjoutErreur(tr("Erreur I/O (initialisation). Aucun périphérique disponible."));
        return false;
    }

    if (bloquant)
    {
        int r = ExecAudio::ExecAudioPCM(donnes.constData(), debut_donnes, fin_donnes, audio->periodSize(), fichier_audio);
        FinLecture_th(r, audio->error());
        return ValideLecture();
    }
    else
    {
        th_lecture = new ExecAudio::ExecAudio_th(donnes.constData(), debut_donnes, fin_donnes, audio->periodSize(), fichier_audio);
        th_lecture->start();
        connect(th_lecture, SIGNAL(FinLecture(int, QAudio::Error)), this, SLOT(FinLecture_th(int, QAudio::Error)));
        return th_lecture->isRunning();
    }
    return false;
}

bool qAudioWave::Nettoyage()
{
    if (!mutex_audio.tryLock(25))
    {
        return false;
    }

    if (th_lecture)
    {
        if (th_lecture->EnCours())
        {
//            qDebug() << "Lecture en cours";
//            mutex_audio.unlock();
//            return false;

            th_lecture->Stop();
//            th_lecture->AttendFinLecture();
            th_lecture->wait(5000);
        }

        delete th_lecture;
        th_lecture = nullptr;
    }

    if (audio)
    {
//        if (audio->error() != QAudio::NoError)
//        {
//            qDebug() << "Cloture lecture avec erreur : " << audio->error();
//        }

        audio->stop();
//        audio->reset();
        delete audio;
        audio = nullptr;
    }

    mutex_audio.unlock();

    return true;
}

void qAudioWave::FinLecture_th(const int etat1, const QAudio::Error etat2)
{
    QThread::msleep(15);

    switch(etat2)
    {
        case QAudio::NoError:
            break;
        case QAudio::OpenError:
            AjoutErreur(tr("Erreur à l'ouverture"));
            break;
        case QAudio::IOError:
            AjoutErreur(tr("Erreur I/O"));
            break;
        case QAudio::UnderrunError:
            AjoutErreur(tr("Erreur de vitesse de lecture"));
            break;
        case QAudio::FatalError:
            AjoutErreur(tr("Erreur Inconnue"));
            break;
        default:
            break;
    }

    switch (etat1)
    {
        case ExecAudio::ERR_OCCUPE:
            AjoutErreur(tr("Périphérique occupé"));
            break;
        case ExecAudio::ERR_IO:
            AjoutErreur(tr("Erreur I/O (écriture)"));
            break;
        case ExecAudio::ERR_TAMPON_VIDE:
            AjoutErreur(tr("Tampon vide"));
            break;
        case ExecAudio::ERR_ARRET:
            AjoutErreur(tr("Arrêt"));
            break;
        default:
            break;
    }

    if (etat1 == ExecAudio::ERR_ARRET)
    {
        valide_lecture = true; // Cas particulier, ne signale pas d'erreur.
    }
    else
    {
        valide_lecture = (etat1 == ExecAudio::ERR_VALIDE) && (etat2 == QAudio::NoError);
    }

    emit FinLecture(valide_lecture);
//    if (valide_lecture)
    {
        Nettoyage();
    }
}
