﻿/**
© Florian Joncour, 2013-2018 florian@zetta-sys.com

Ce logiciel est un programme informatique faisant interface à la bibliothèque
Perspective3D, un outil de modélisation 3D à partir de vues orthographiques 2D.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site "http://www.cecill.info".

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée.  Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme,  le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard  l'attention de l'utilisateur est attirée sur les risques
associés au chargement,  à l'utilisation,  à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant  des  connaissances  informatiques approfondies.  Les
utilisateurs sont donc invités à charger  et  tester  l'adéquation  du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.
**/

#include "gestionnairescripts.h"

#include "mainwindow.h"
#include "animeQt.h"
#include "automate.h"
#include <QFile>
#include <QDir>
#include <QListWidget>
#include <QDesktopWidget>
#include <QApplication>
#include "API/Qt/Conversion.h"
#include "API/Qt/Utils.h"

#ifdef DEBUG
#include <QDebug>
#endif

#define TAILLE_FENETRE_X 600
#define TAILLE_FENETRE_Y 220

GestionnaireScripts::GestionnaireScripts(const AnimationQt &anim, MainWindow *parent) : CadreDialog(parent), barre_fenetre(0), animation(anim)
{
    Parent = parent;
    ui.setupUi(this);

    chrono.start();

    setMaximumWidth(TAILLE_FENETRE_X);
    setMinimumWidth(TAILLE_FENETRE_X);
    setMinimumHeight(TAILLE_FENETRE_Y);
    setMaximumHeight(TAILLE_FENETRE_Y);
    setWindowTitle(tr("Scripts"));

    ui.lw_liste_scripts->setVisible(true);
    ui.lw_liste_source->setVisible(false);

    barre_fenetre = new BarreFenetre(this); /* A la fin pour le retrouver au dessus de la pile d'affichage. */
//    layout()->addWidget(barre_fenetre);
    barre_fenetre->defTitre(tr("Scripts"));
//    PlacementFenetre();
}

GestionnaireScripts::~GestionnaireScripts()
{
    if (barre_fenetre)
    {
        delete barre_fenetre;
    }
}

void GestionnaireScripts::focusOutEvent(QFocusEvent *ev)
{
    close();
    ev->accept();
}

void GestionnaireScripts::showEvent(QShowEvent *ev)
{
    emit EvGestionScript(true);
    ui.lw_liste_scripts->setVisible(true);
    PlacementFenetre();
    ev->accept();
}

void GestionnaireScripts::closeEvent(QCloseEvent *ev)
{
    emit EvGestionScript(false);
    ev->accept();
}

void GestionnaireScripts::resizeEvent(QResizeEvent *ev)
{
    if (barre_fenetre)
    {
        barre_fenetre->Redim(ev->size());
    }
    else
    {
    }
    ev->accept();
}

void GestionnaireScripts::PlacementFenetre()
{
//    move(0,0);
//    return;

    QScreen *ecran = EcranWidget(Parent);

//    const QPoint position_curseur = PositionCurseurWidget(Parent);
//    const int marge_x = position_curseur.isNull() ? 75 : position_curseur.x()+75;
//    const int marge_y = position_curseur.isNull() ? 45 : position_curseur.x()+45;

    const int marge_x = 15;
    const int marge_y = 45;

    if (ecran)
    {
        QRect rect = ecran->geometry();
//        move(marge_x, rect.height()-TAILLE_FENETRE_Y-marge_y);
        move((rect.width()-(TAILLE_FENETRE_X+marge_x)), rect.height()-TAILLE_FENETRE_Y-marge_y);
    }
    else
    {
        QRect rect = QApplication::desktop()->screenGeometry(QApplication::desktop()->screenNumber(Parent));
        move((rect.width())-(TAILLE_FENETRE_X+marge_x), rect.height()-TAILLE_FENETRE_Y-marge_y);
    }
}

void GestionnaireScripts::AfficheScripts(const QString &dossier, const QString &fichier_plan)
{
    show();
    PlacementFenetre();
    ui.lw_liste_scripts->setVisible(true);

    dossier_scripts = dossier;
    fichier_courant = fichier_plan;
//    chrono.restart();

    if (!dossier_scripts.isEmpty())
    {
        QDir dir(dossier_scripts);

        ui.lw_liste_scripts->clear();
        ui.lw_liste_source->clear();
        QStringList liste_fichiers = dir.entryList(QStringList(QString("*.") + ExtensionScriptsAutomate), QDir::Files, QDir::Time);

        if (!fichier_plan.isEmpty())
        {
            QString n_fichier_plan = FichierChemin(fichier_courant);

            for(int i=0; i<liste_fichiers.size(); ++i)
            {
                const QString &s = liste_fichiers[i];
                if (s.contains(n_fichier_plan, Qt::CaseInsensitive)) /* Sélection des fichiers attachés au plan courant. */
                {
                    ui.lw_liste_scripts->addItem(liste_fichiers[i]);
                }
            }
        }
        else
        {
            for(int i=0; i<liste_fichiers.size(); ++i)
            {
                ui.lw_liste_scripts->addItem(liste_fichiers[i]);
            }
        }
    }
    animation.DeplaceWidgetCurseur(this);
}

QString GestionnaireScripts::ScriptCourant(const QString &s) const
{
    if (!dossier_scripts.isEmpty())
    {
        return dossier_scripts + QDir::separator() + s;
    }
    else
    {
        return s;
    }
}

void GestionnaireScripts::on_lw_liste_scripts_itemClicked(QListWidgetItem *item)
{
    QString chemin_script = ScriptCourant(item->text());
    OuvreScript(chemin_script);
}

bool GestionnaireScripts::OuvreScript(const QString &chemin_script)
{
    if (!isVisible())
        show();

    chrono.restart();
    ui.lw_liste_source->clear();
    ui.lw_liste_source->setVisible(true);

    ui.lbl_txt->setText(FichierChemin(chemin_script));

    if (QFile::exists(chemin_script))
    {
        QFile f_scr(chemin_script);
        if (f_scr.open(QFile::ReadOnly | QFile::Text))
        {
            QString txt = QString::fromUtf8(f_scr.readAll());
            f_scr.close();

            QStringList ls_txt = txt.split(QRegExp("[\r\n]"), QString::KeepEmptyParts);
            if (!txt.isEmpty())
            {
                for(int i=0; i<ls_txt.size(); ++i)
                {
                    if (i==ls_txt.size()-1)
                    {
                        if (ls_txt[i].isEmpty()) /* On triche pour passer la denière ligne si elle est vierge */
                            break;
                    }
                    ui.lw_liste_source->addItem(ls_txt[i]);
                }
                return true;
            }
        }
    }
    return false;
}

bool GestionnaireScripts::AfficheLigne(int id)
{
    if (isVisible() && (id >= 0) && (id < ui.lw_liste_source->count()))
    {
        if (ui.lw_liste_scripts->isVisible())
            ui.lw_liste_scripts->setVisible(false);
        ui.lw_liste_source->setCurrentRow(id);

        QListWidgetItem *it = ui.lw_liste_source->item(id);
        if (it)
        {
            ui.lw_liste_source->scrollToItem(it, QAbstractItemView::PositionAtCenter);
        }

        return true;

//        QListWidgetItem *it = ui.lw_liste_source->item(id);
//        if (it)
//        {
//            it->setSelected(true);
//            return true;
//        }
    }
    return false;
}

void GestionnaireScripts::on_lw_liste_scripts_itemDoubleClicked(QListWidgetItem *item)
{
    if (Parent->Ouvre_Script(ScriptCourant(item->text())))
    {
        ui.lw_liste_scripts->setVisible(false);
        ui.lw_liste_source->setVisible(true);
    }
}

void GestionnaireScripts::on_lw_liste_source_currentRowChanged(int currentRow)
{
    if (currentRow < 0)
    {
        ui.lbl_id_ligne->clear();
    }
    else
    {
//        const int pct = int(float(currentRow+1) * (100.f / float(ui.lw_liste_source->count())));

        QTime chrono_d(0,0,0,0);
        chrono_d = chrono_d.addMSecs(chrono.elapsed());

        if (!chrono_d.isNull())
        {
            ui.lbl_id_ligne->setText(QString::number(currentRow+1) + " / " + QString::number(ui.lw_liste_source->count()) + "   (" + chrono_d.toString("m:ss") + ")");
        }
        else
        {
            ui.lbl_id_ligne->setText(QString::number(currentRow+1) + " / " + QString::number(ui.lw_liste_source->count()));
        }
    }
}
