﻿/**
© Florian Joncour, 2013-2018 florian@zetta-sys.com

Ce logiciel est un programme informatique faisant interface à la bibliothèque
Perspective3D, un outil de modélisation 3D à partir de vues orthographiques 2D.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site "http://www.cecill.info".

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée.  Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme,  le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard  l'attention de l'utilisateur est attirée sur les risques
associés au chargement,  à l'utilisation,  à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant  des  connaissances  informatiques approfondies.  Les
utilisateurs sont donc invités à charger  et  tester  l'adéquation  du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.
**/

extern "C"
{
    #include "libttspico/picoapi.h"
    #include "libttspico/picoapid.h"
    #include "libttspico/picoos.h"
}

#include <string.h>
#include <QDir>
#include <QThread>
#include <QTime>
#include <QAudioDeviceInfo>
#include <QCoreApplication>
#include <QEventLoop>
#include "qAudioWave.h"
#include "API/Qt/Utils.h"

void SommeilTempoMs(int ms)
/* Sommeil non bloquant pour l'interface (bloque l'exécution mais pas l'affichage pendant un temps donné en millisecondes) */
{
#if 0
//    QThread::msleep(ms); // Bloque l'interface.
    SommeilTempoQt(ms);
#else
    if (ms < 1)
    {
        return;
    }
    const QTime temps_sommeil = QTime::currentTime().addMSecs(ms);
    while(QTime::currentTime() < temps_sommeil)
    {
        QCoreApplication::processEvents(QEventLoop::AllEvents, 100);
        QThread::msleep(5);
    }
#endif // 1
}

#ifdef DEBUG
#include <QDebug>
#endif

// #include <QRegExp>
// #include <QStringList>

#include <QByteArray>
#include <QIODevice>
#include <QAudioBuffer>
#include <QAudioOutput>

#include "qDicte.h"

/* Configuration: */
#define TAILLE_TAMPON_PICO       2500000
#define CHEMIN_LANG "../langPico/" /* Répertoire des données des langues */
#define PICO_VOICE "PicoVoice"

const char *picoInternalTaLingware[] = { CHEMIN_LANG "fr-FR_ta.bin",
                                            CHEMIN_LANG "en-US_ta.bin",
                                            CHEMIN_LANG "en-GB_ta.bin",
                                            CHEMIN_LANG "es-ES_ta.bin",
                                            CHEMIN_LANG "it-IT_ta.bin",
                                            CHEMIN_LANG "de-DE_ta.bin" };

const char *picoInternalSgLingware[] = { CHEMIN_LANG "fr-FR_nk0_sg.bin",
                                            CHEMIN_LANG "en-US_lh0_sg.bin",
                                            CHEMIN_LANG "en-GB_kh0_sg.bin",
                                            CHEMIN_LANG "es-ES_zl0_sg.bin",
                                            CHEMIN_LANG "it-IT_cm0_sg.bin",
                                            CHEMIN_LANG "de-DE_gl0_sg.bin" };

static bool InitInst = true; /* Pour n'autoriser qu'une seule instance à la fois. */

pico_System     picoSystem          = nullptr;
pico_Resource   picoTaResource      = nullptr;
pico_Resource   picoSgResource      = nullptr;
pico_Engine     picoEngine          = nullptr;
pico_Char *     picoTaResourceName  = nullptr;
pico_Char *     picoSgResourceName  = nullptr;

qDicte::qDicte(int langue, const QString &racine_ressources) :
    RacineRessources(racine_ressources), valide_init(false), force_arret(false), diction_en_cours(false), langue_pico(langue),
    tampon_pico(0), volume_audio(0.8), audio(nullptr), fichier_audio(nullptr), peripherique_audio(nullptr)
{
    valide_init = (langue < LANG_NUL && langue >= 0) ? InitPico(langue) : false;
}

qDicte::~qDicte()
{
//    if (audio)
//        { delete audio; }
    TerminePico();
}

bool qDicte::Reinit(int langue)
{
    message_erreur.clear();

    if (valide_init)
    {
        TerminePico();
    }

    if (langue >= LANG_NUL || langue < 0)
    {
        return false;
    }

    force_arret = false;
    diction_en_cours = false;
    valide_init = InitPico(langue);
    return valide_init;
}

bool qDicte::InitPico(int langue)
/* Initialisation d'une session pico. */
{
    if (!InitInst)
    {
        return false;
    }

    langue_pico = langue;

    if (RacineRessources.isEmpty())
    {
        RacineRessources = ".";
    }

    int etat;
    pico_Retstring message;

    tampon_pico = new char[TAILLE_TAMPON_PICO];

    /* Initialisation */
    etat = pico_initialize((void *) tampon_pico, TAILLE_TAMPON_PICO, &picoSystem);
    if (etat)
    {
        pico_getSystemStatusMessage(picoSystem, etat, message);
        AjoutErreur(tr("Impossible d'initialiser pico") + " : " + message);
        TerminePico(true);
        return false;
    }

    /* Charge le fichier d'analyse du texte */
    QString chemin_TA = RacineRessources + QDir::separator() + picoInternalTaLingware[langue];
    etat = pico_loadResource(picoSystem, (const pico_Char *) chemin_TA.toStdString().data(), &picoTaResource);
    if (etat)
    {
        pico_getSystemStatusMessage(picoSystem, etat, message);
        AjoutErreur(tr("Impossible de charger le fichier (analyse)") + " : " + message);
        TerminePico(true);
        return false;
    }

    /* Charge le fichier de génération du signal */
    QString chemin_SG = RacineRessources + QDir::separator() + picoInternalSgLingware[langue];
    etat = pico_loadResource(picoSystem, (const pico_Char *) chemin_SG.toStdString().data(), &picoSgResource);
    if (etat)
    {
        pico_getSystemStatusMessage(picoSystem, etat, message);
        AjoutErreur(tr("Impossible de charger le fichier (signal)") + " : " + message);
        TerminePico(true);
        return false;
    }

    picoTaResourceName = (pico_Char *) new char[PICO_MAX_RESOURCE_NAME_SIZE];
    etat = pico_getResourceName(picoSystem, picoTaResource, (char *) picoTaResourceName);
    if (etat)
    {
        pico_getSystemStatusMessage(picoSystem, etat, message);
        AjoutErreur(tr("Impossible d'obtenir la ressource d'analyse de texte") + " : " + message);
        TerminePico(true);
        return false;
    }

    picoSgResourceName = (pico_Char *) new char[PICO_MAX_RESOURCE_NAME_SIZE];
    etat = pico_getResourceName(picoSystem, picoSgResource, (char *) picoSgResourceName);
    if (etat)
    {
        pico_getSystemStatusMessage(picoSystem, etat, message);
        AjoutErreur(tr("Impossible d'obtenir la ressource de génération du signal") + " : " + message);
        TerminePico(true);
        return false;
    }

    /* Créé la voix */
    etat = pico_createVoiceDefinition(picoSystem, (const pico_Char *) PICO_VOICE);
    if (etat)
    {
        pico_getSystemStatusMessage(picoSystem, etat, message);
        AjoutErreur(tr("Impossible de générer la voix") + " : " + message);
        TerminePico(true);
        return false;
    }

    etat = pico_addResourceToVoiceDefinition(picoSystem, (const pico_Char *) PICO_VOICE, picoTaResourceName);
    if (etat)
    {
        pico_getSystemStatusMessage(picoSystem, etat, message);
        AjoutErreur(tr("Impossible d'ajouter l'analyseur de texte à la voix") + " : " + message);
        TerminePico(true);
        return false;
    }

    etat = pico_addResourceToVoiceDefinition(picoSystem, (const pico_Char *) PICO_VOICE, picoSgResourceName);
    if (etat)
    {
        pico_getSystemStatusMessage(picoSystem, etat, message);
        AjoutErreur(tr("Impossible d'ajouter le générateur de signal à la voix") + " : " + message);
        TerminePico(true);
        return false;
    }

    etat = pico_newEngine(picoSystem, (const pico_Char *) PICO_VOICE, &picoEngine);
    if (etat)
    {
        pico_getSystemStatusMessage(picoSystem, etat, message);
        AjoutErreur(tr("Impossible de créer le moteur TTS") + " : " + message);
        TerminePico(true);
        return false;
    }

    InitInst = false;
    return true;
}

bool qDicte::TerminePico(bool force)
/* Termine une session pico. */
{
    if (InitInst && !force)
        return false;

    if (picoEngine)
    {
        pico_disposeEngine(picoSystem, &picoEngine);
        pico_releaseVoiceDefinition(picoSystem, (pico_Char *) PICO_VOICE);
        picoEngine = nullptr;
    }

    if (picoTaResource)
    {
        pico_unloadResource(picoSystem, &picoTaResource);
        picoTaResource = nullptr;
    }

    if (picoSgResource)
    {
        pico_unloadResource(picoSystem, &picoSgResource);
        picoSgResource = nullptr;
    }

    if (picoSystem)
    {
        pico_terminate(&picoSystem);
        picoSystem = nullptr;
    }

    if (tampon_pico)
    {
        delete [] tampon_pico;
        tampon_pico = nullptr;
    }

    if (picoTaResourceName)
    {
        delete [] picoTaResourceName;
        picoTaResourceName = nullptr;
    }

    if (picoSgResourceName)
    {
        delete [] picoSgResourceName;
        picoSgResourceName = nullptr;
    }

    InitInst = true;
    return true;
}

// bool qDicte::Dicte(const char *texte, unsigned int taille_texte)
// {

// }

bool qDicte::defPeripherique(QAudioDeviceInfo const * peripherique)
{
    peripherique_audio = peripherique;
    return true;
}


void qDicte::stopDiction()
/* Arrêt de la diction. */
{
    if (diction_en_cours)
    {
        force_arret = true;
        while (diction_en_cours)
        {
            QThread::msleep(20);
        }
        QThread::msleep(50);
    }
    force_arret = false;
}

bool qDicte::Dicte(const QString &texte, const QAudioDeviceInfo *peripherique)
{
    if (diction_en_cours)
    {
        AjoutErreur(tr("Une diction est déjà en cours..."));
        return false;
    }
    if (valide_init)
    {
        force_arret = false;
        diction_en_cours = true;
        message_erreur.clear();

        QString copie(texte);
        copie.replace('!', " ! ");
        copie.replace('!', " . ");
        copie.replace('?', " ? ");
        copie.remove('\''); /* Il y a parfois des erreurs de prononciation genre avec "n'hésitez pas" */
//        copie.replace('+', " + ");
//        copie.replace('-', " - ");
//        copie.replace('*', " * ");
//        copie.replace('/', " / ");
        copie += "                                                  ."; /* Laisse de la marge à la fin du tampon, Pico a parfois du mal à terminer ses phrases... */

        const QAudioDeviceInfo &perif = (!peripherique) ? (!peripherique_audio ? QAudioDeviceInfo() : *peripherique_audio) : *peripherique;
        if (perif.isNull())
        {
            AjoutErreur(tr("Périphérique invalide"));
            return false;
        }

        bool valide_exec = Synthese((const char *) copie.toStdString().data(), copie.size(), true, perif);
        diction_en_cours = false;
        audio = nullptr;
        return valide_exec;

//     QStringList ls_txt = texte.split(QRegExp("[.!?]"));
//         const unsigned int n_ls = ls_txt.size();
//         for(unsigned int i=0; i<n_ls; ++i)
//         {
//             qDebug() << ":: " << ls_txt[i];
//             if (!Synthese((const char *) ls_txt[i].toStdString().data(), ls_txt[i].size(), (i==0)))
//             {
//                 return false;
//             }
//
// //             qDebug() << "Sommeil (avant)...";
// //             SommeilTempoMs(800); /* Délai entre chaque phrases pour *ponctuer* la diction. */
// //             qDebug() << "Sommeil (après)...";
//         }
//         return true;
    }
    AjoutErreur(tr("Moteur non initialisé."));
    return false;
}

bool qDicte::Synthese(const char *texte, unsigned int taille_texte, bool nouvelle_synth, const QAudioDeviceInfo &peripherique)
{
    if (!taille_texte)
    {
        return true;
    }

    if (nouvelle_synth)
    {
        pico_resetEngine(picoEngine, PICO_RESET_SOFT); // PICO_RESET_FULL
    }

    pico_Char *local_text = (pico_Char *) texte;
    pico_Char *inp = (pico_Char *) local_text;
    int text_remaining = taille_texte+1;

    int etat = 0;
    int etat_pico = 0;

    const unsigned int taille_tampon_pico = 256;
    char tampon_pico[taille_tampon_pico+1];
    pico_Retstring message = { '\0' };

    pico_Int16 bytes_sent = 0;
    pico_Int16 bytes_sent_nil = 0;
    pico_Int16 bytes_recv = 0;
    pico_Int16 out_data_type = 0;

    QByteArray tampon_audio;

    while (text_remaining) /* Génère la synthèse vocale. */
    {
        if ((etat = pico_putTextUtf8(picoEngine, inp, text_remaining, &bytes_sent))) /* Envoi du texte au moteur SVOX */
        {
            pico_getSystemStatusMessage(picoSystem, etat, message);
            AjoutErreur(tr("Impossible d'ajouter le texte dans le moteur") + " : " + message);
            return false;
        }

#ifdef DEBUG
        qDebug() << "TTS : \"" << (const char *) inp << "\" : " << bytes_sent << "/" << text_remaining;
#endif // DEBUG
        text_remaining -= bytes_sent;
        inp += bytes_sent;

        if (!text_remaining)
        {
            pico_putTextUtf8(picoEngine, (pico_Char*)"\0", 1, &bytes_sent_nil); /* Pour assurer au moteur pico la fin du contexte de diction. */
        }

//        qDebug() << "Debut envoi des données...";

        do
        {
            etat_pico = pico_getData(picoEngine, (void *) tampon_pico, taille_tampon_pico, &bytes_recv, &out_data_type); /* Recupère les données. */

            if (etat_pico == PICO_STEP_BUSY || etat_pico == PICO_STEP_IDLE)
            {
                if (bytes_recv)
                {
                    tampon_audio.append(tampon_pico, bytes_recv);
                }
                else /* Pico cherche sa voix/voie... */
                {
//                    SommeilTempoMs(10);
                }
            }
            else /* Erreur */
            {
                pico_getSystemStatusMessage(picoSystem, etat_pico, message);
                AjoutErreur(tr("Impossible d'obtenir les données du moteur") + " : " + message);
#ifdef DEBUG
                qDebug() << "qDicte :: Erreur (Pico)";
#endif // DEBUG
                return false;
            }

            if (force_arret)
            {
#ifdef DEBUG
                qDebug() << "qDicte :: Arrêt !";
#endif // DEBUG
                AjoutErreur(tr("Arrêt"));
                return true;
            }

        } while (etat_pico != PICO_STEP_IDLE);

        if (force_arret)
        {
#ifdef DEBUG
            qDebug() << "qDicte :: Arrêt !";
#endif // DEBUG
            AjoutErreur(tr("Arrêt"));
            return true;
        }
    }

    if (!tampon_audio.size())
    {
#ifdef DEBUG
        qDebug() << "qDicte :: Rien à dicter pour \"" << texte << "\" ?! :: " << tampon_audio.size();
#endif
        return true; /* C'est pas forcément une erreur en soit... */
    }

#ifdef DEBUG
    qDebug() << "Envoi des données audio (" << tampon_audio.size() << ")";
#endif

    QAudioFormat format;
    format.setSampleRate(15500); // 16000
    format.setChannelCount(1);
    format.setSampleSize(16);
    format.setCodec("audio/pcm");
    format.setByteOrder(QAudioFormat::LittleEndian);
    format.setSampleType(QAudioFormat::SignedInt);

    if (peripherique.isNull())
    {
        AjoutErreur(tr("Périphérique audio non défini."));
        return false;
    }

//#ifdef DEBUG
    if (!peripherique.isFormatSupported(format))
    {
        AjoutErreur(tr("Format audio de sortie non supporté par le périphérique.\nLe périphérique est peut être occupé."));
        return false;
    }
//#endif

    audio = new QAudioOutput(peripherique, format);

    // Test:
    fichier_audio = audio->start();
    if (audio->state() != QAudio::ActiveState && audio->state() != QAudio::IdleState)
    {
        AjoutErreur(tr("Erreur I/O (initialisation). Impossible de démarrer le périphérique"));
        return false;
    }

    if (!fichier_audio)
    {
//         qDebug() << "Erreur lors de l'ouverture du périphérique ?!";
        AjoutErreur(tr("Erreur I/O (initialisation). Aucun périphérique disponible."));
        return false;
    }

    audio->setVolume(volume_audio);
    const int r = ExecAudio::ExecAudioPCM(tampon_audio.constData(), 0, tampon_audio.size(), audio->periodSize(), fichier_audio, &force_arret);
    const QAudio::Error r2 = audio->error();
    fichier_audio->close();
    audio->stop();
    delete audio;
    audio = nullptr;

    switch(r2)
    {
        case QAudio::NoError:
            break;
        case QAudio::OpenError:
            AjoutErreur(tr("Erreur à l'ouverture"));
            break;
        case QAudio::IOError:
            AjoutErreur(tr("Erreur I/O"));
            break;
        case QAudio::UnderrunError:
            AjoutErreur(tr("Erreur de vitesse de lecture"));
            break;
        case QAudio::FatalError:
            AjoutErreur(tr("Erreur Inconnue"));
            break;
        default:
            break;
    }

    switch (r)
    {
        case ExecAudio::ERR_OCCUPE:
            AjoutErreur(tr("Périphérique occupé"));
            break;
        case ExecAudio::ERR_IO:
            AjoutErreur(tr("Erreur I/O (écriture)"));
            break;
        case ExecAudio::ERR_TAMPON_VIDE:
            AjoutErreur(tr("Tampon vide"));
            break;
        case ExecAudio::ERR_ARRET:
            AjoutErreur(tr("Arrêt"));
            break;
        default:
            break;
    }

#ifdef DEBUG
    if (r != ExecAudio::ERR_VALIDE)
    {
        qDebug() << "qDicte::Synthese() :: Retour ExecAudioPCM() : " << r;
        qDebug() << "Erreurs : " << MessageErreur();
    }
#endif

    if (force_arret || r == ExecAudio::ERR_ARRET)
    {
        return true; /* Cas particulier, on ne signal pas en tant qu'erreur. */
    }

    return (r == ExecAudio::ERR_VALIDE) && (r2 == QAudio::NoError);
}

Th_qDicte::Th_qDicte(int langue, const QString &racine_ressources) : texte_tts(""), peripherique_tts(0), retour_diction(false), diction(langue, racine_ressources)
{
    ;
}

Th_qDicte::~Th_qDicte()
{
    ;
}

bool Th_qDicte::configurationDiction(int langue, qreal volume, const QAudioDeviceInfo *peripherique, const QString &chemin_ressources)
{
    if (diction_en_cours)
    {
        return false;
    }

    if (!chemin_ressources.isEmpty())
    {
        diction.defCheminRessources(chemin_ressources); /* Impérativement avant l'init de Pico ! */
    }

    if (langue != diction.Langue())
    {
        diction.Reinit(langue);
    }

    if (!qFuzzyCompare(diction.VolumeAudio()+1, volume+1))
    {
        diction.defVolumeAudio(volume);
    }

    if (peripherique)
    {
        if (peripherique != peripherique_tts && (!peripherique->isNull()))
        {
            peripherique_tts = peripherique;
        }
    }
    return true;
}

bool Th_qDicte::execDiction(const QString &texte, int langue, qreal volume, const QAudioDeviceInfo *peripherique)
{
    if (!diction.Valide() || texte.isEmpty())
    {
        return false;
    }

    if (diction.DictionEnCours() || diction_en_cours)
    {
        diction.AjoutErreur(tr("Une diction est déjà en cours."));
        return false;
    }

    if (!mutex.tryLock(500))
    {
        return true;
    }

    configurationDiction(langue, volume, peripherique, QString());

    texte_tts = texte;

    diction_en_cours = true;
    retour_diction = false;
    start(); /* Exécution de la diction. */

    do  /* Attend la fin de la diction (sans bloquer l'interface). */
    {
        SommeilTempoMs(10); // Un mutex est bloqué dans Qt si appelé conjointement à la fonction SommeilTempoQt() de AnimeQt
    } while (diction_en_cours);

//    qDebug() << "FIN (1) : " << n_cpt;
    mutex.unlock();
//    qDebug() << "FIN (2) : " << n_cpt;
    return retour_diction;
}

void Th_qDicte::stopDiction()
{
    if (diction_en_cours)
    {
//        mutex.unlock();
        retour_diction = true;
        diction.stopDiction();

        while (diction_en_cours) /* Attend la fin de la diction (sans bloquer l'interface). */
        {
            QThread::msleep(50);
        }
    }
}

void Th_qDicte::run()
{
    retour_diction = diction.Dicte(texte_tts, peripherique_tts);
    diction_en_cours = false;
}
