﻿/**
© Florian Joncour, 2013-2018 florian@zetta-sys.com

Ce logiciel est un programme informatique faisant interface à la bibliothèque
Perspective3D, un outil de modélisation 3D à partir de vues orthographiques 2D.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site "http://www.cecill.info".

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée.  Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme,  le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard  l'attention de l'utilisateur est attirée sur les risques
associés au chargement,  à l'utilisation,  à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant  des  connaissances  informatiques approfondies.  Les
utilisateurs sont donc invités à charger  et  tester  l'adéquation  du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.
**/

/* Interface SVox Pico -> Qt pour la diction TTS. */

#ifndef QDICTION_H
#define QDICTION_H

#include <QObject>
#include <QString>
#include <QThread>
#include <QMutex>

class QIODevice;
class QAudioOutput;
class QAudioDeviceInfo;

class qDicte : public QObject
{
    Q_OBJECT

        friend class Th_qDicte;

    public:
        enum LANG { LANG_FR=0, LANG_EN_US, LANG_EN_GB, LANG_ES, LANG_IT, LANG_DE, LANG_NUL }; /* LANG_NUL obligatoirement en dernier. */

        qDicte(int langue=LANG_NUL, const QString &racine_ressources=QString("."));
        ~qDicte();

        bool Reinit(int langue);
        /* Réinitialisation du moteur avec la langue assignée. */

        bool Dicte(const QString &texte, const QAudioDeviceInfo *peripherique=0);
        /* Dicte le texte donné en argument */

        void stopDiction();
        /* Arrêt de la diction. */

        bool Valide() const { return valide_init; }
        /* Contrôle si le module TTS et prêt à la diction. */

        int Langue() const { return langue_pico; }

        inline bool DictionEnCours() const { return diction_en_cours; }

        const QString &MessageErreur() const { return message_erreur; }
        /* Renvoi le dernier message d'erreur. */

        inline void defVolumeAudio(qreal v)
        /* Assigne le volume audio (de 0 à 1). */
        {
            volume_audio = v;
        }

        inline void defCheminRessources(const QString &chemin)
        /* Assigne le volume audio (de 0 à 1). */
        {
            RacineRessources = chemin;
        }

        inline qreal VolumeAudio() const
        /* Renvoi le volum audio. */
        {
            return volume_audio;
        }

        bool defPeripherique(QAudioDeviceInfo const * peripherique);

    protected:
        bool InitPico(int langue);
        bool TerminePico(bool force=false);
        bool Synthese(const char *texte, unsigned int taille_texte, bool nouvelle_synth, const QAudioDeviceInfo &peripherique);

        inline void AjoutErreur(const QString &txt) { message_erreur += (txt + QString("\n")); }

        QString RacineRessources;
        bool valide_init;
        bool force_arret;
        bool diction_en_cours;
        int langue_pico;
        char *tampon_pico;
        qreal volume_audio;
        QAudioOutput *audio;
        QIODevice *fichier_audio;
        QString chemin_fichier_audio;
        QString message_erreur;
        QAudioDeviceInfo const * peripherique_audio;
};

class Th_qDicte : public QThread
{
        /* Diction dans un fil d'exécution séparé pour ne pas bloquer l'interface graphique. */
        Q_OBJECT
    public:
        Th_qDicte(int langue=qDicte::LANG_NUL, const QString &racine_ressources=QString("."));
        ~Th_qDicte();

        bool configurationDiction(int langue, qreal volume, const QAudioDeviceInfo *peripherique, const QString &chemin_ressources=QString());
        bool execDiction(const QString &texte, int langue, qreal volume=0.75, QAudioDeviceInfo const * peripherique=0);
        void stopDiction();

        inline const qDicte &Diction() const { return diction; }

        inline bool DictionEnCours() const { return diction_en_cours; }

    protected:
        void run();

        QString texte_tts;
        QAudioDeviceInfo const *peripherique_tts;
        QMutex mutex;
        bool diction_en_cours;
        bool retour_diction;
        qDicte diction;
};

#endif /* QDICTION_H */
